package ru.yandex.chemodan.app.notifier.push.filter;

import org.jetbrains.annotations.NotNull;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.push.body.XivaPushSpecificParamsCreator;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannelGroup;
import ru.yandex.chemodan.xiva.XivaEvent;
import ru.yandex.chemodan.xiva.XivaEventRecipient;
import ru.yandex.chemodan.xiva.XivaSubscription;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class NotificationPushFilterManagerTest {
    private static final XivaSubscription SUBSCRIPTION_TO_BRIGHT1 =
            consSubscriptionWithFilter("1", "uuid#1", "ios_bright");

    private static final XivaSubscription SUBSCRIPTION_TO_BRIGHT4 =
            consSubscriptionWithFilter("4", "uuid#4", "ios_bright");

    private static final ListF<XivaSubscription> SUBSCRIPTIONS = Cf.list(
            SUBSCRIPTION_TO_BRIGHT1,
            consSubscriptionWithFilter("2", "uuid#2", "basic filter"),
            consSubscription("3", "uuid#3"),
            SUBSCRIPTION_TO_BRIGHT4
    );

    @Test
    public void allEnabledForNonBrightGroup() {
        assertEnabled(GlobalSubscriptionChannelGroup.WEB_DESKTOP_V1, "enabled-group", "enabled-action");
        assertEnabled(GlobalSubscriptionChannelGroup.MOBILE_V1, "enabled-group", "enabled-action");
        assertEnabled(GlobalSubscriptionChannelGroup.MOBILE_V2, "enabled-group", "enabled-action");
        assertEnabled(GlobalSubscriptionChannelGroup.ALL, "enabled-group", "enabled-action");
    }

    @Test
    public void explicitlyEnabled() {
        assertSubs("enabled-group", "enabled-action", SUBSCRIPTION_TO_BRIGHT1, SUBSCRIPTION_TO_BRIGHT4);

    }

    @Test
    public void explicitlyDisabled() {
        assertSubs("disabled-group", "disabled-action");

    }

    @Test
    public void otherSomeEnabledSomeDisabled() {
        assertSubs("other-group", "other-action", SUBSCRIPTION_TO_BRIGHT1);
    }

    @Test
    public void hasValidBrightPushImpl() {
        String filter = "{\"rules\":[{\"do\":\"send_bright\",\"if\":\"CommonEvents & Tags\"},{\"do\":\"send_bright\",\"if\":\"CommonEvents & BrightEnabled\"},{\"do\":\"send_bright\",\"if\":\"CommonEvents & DeviceIdTag\"},{\"do\":\"send_bright\",\"if\":\"CommonEvents & BrightEnabled & Version\"},{\"do\":\"skip\"}],\"vars\":{\"DeviceIdTag\":{\"$has_tags\":[\"device_id.b02f000f118945fab92b8db5affdadf8\"]},\"BrightEnabled\":{\"$has_tags\":[\"ios_bright\"]},\"Version\":{\"$has_tags\":[\"2.62\"]},\"CommonEvents\":{\"$event\":[\"notification_mobile_v2\"]},\"Tags\":{\"$has_tags\":[\"ios\"]}}}";

        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("2.61-"))));
        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("2.62-"))));
        Assert.isFalse(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("2.63-"))));

        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("-2.62"))));
        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("-2.63"))));
        Assert.isFalse(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.of(XivaPushSpecificParamsCreator.parseAppVersionsInterval("-2.61"))));

        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(filter), Option.empty()));
        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of("{}"), Option.empty()));
        Assert.isTrue(NotificationPushFilterManager.hasValidBrightPushImpl(Option.of(""), Option.empty()));
    }

    private static void assertSubs(String group, String action, XivaSubscription... enabledSubs) {
        assertResultEquals(
                GlobalSubscriptionChannelGroup.MOBILE_V2_BRIGHT, group, action,
                NotificationPushFilter.includeSubscriptionIds(Cf.list(enabledSubs).map(XivaSubscription::getId))
        );
    }

    @SuppressWarnings("SameParameterValue")
    private static void assertEnabled(GlobalSubscriptionChannelGroup channelGroup, String group, String action) {
        assertResultEquals(channelGroup, group, action, NotificationPushFilter.ALL_ENABLED);
    }

    private static void assertResultEquals(GlobalSubscriptionChannelGroup channelGroup, String group, String action,
                                           NotificationPushFilter expected)
    {
        MapF<String, DevicePushFilterConfig> map = Cf.map(
                SUBSCRIPTION_TO_BRIGHT1.getUuid().get(), consPushFilter(true),
                SUBSCRIPTION_TO_BRIGHT4.getUuid().get(), consPushFilter(false)
        );
        Assert.equals(expected,
                NotificationPushFilterManager.getFilter(SUBSCRIPTIONS, map, channelGroup, group, action,
                        new XivaEvent(XivaEventRecipient.passport(1), "")
                )
        );
    }

    @NotNull
    private static DevicePushFilterConfig consPushFilter(boolean otherEnabled) {
        DevicePushFilterConfig.GroupAndActionPattern enabledGroupAndAction =
                new DevicePushFilterConfig.GroupAndActionPattern("enabled-group", "enabled-action");
        DevicePushFilterConfig.GroupAndActionPattern disabledGroupAndAction =
                new DevicePushFilterConfig.GroupAndActionPattern("disabled-group", "disabled-action");
        return new DevicePushFilterConfig(Cf.list(
                new DevicePushFilterConfig.Matcher("enabled-grp-act", Cf.list(enabledGroupAndAction),
                        true),
                new DevicePushFilterConfig.Matcher("disabled-grp-act", Cf.list(disabledGroupAndAction),
                        false),
                new DevicePushFilterConfig.Matcher("other", Cf.list(), otherEnabled)
        ));
    }

    @SuppressWarnings("SameParameterValue")
    private static XivaSubscription consSubscription(String id, String uuid) {
        return XivaSubscription.builder()
                .id(id)
                .uuid(uuid)
                .build();
    }

    private static XivaSubscription consSubscriptionWithFilter(String id, String uuid, String filter) {
        return XivaSubscription.builder()
                .id(id)
                .filter(filter)
                .uuid(uuid)
                .build();
    }
}
