package ru.yandex.chemodan.app.notifier.rtx;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import NRtx.NProto.Api;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import lombok.Data;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils2;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class RtxClient {
    private static final Logger logger = LoggerFactory.getLogger(RtxClient.class);

    private final HttpClient httpClient;
    private final String baseUri;

    public RtxClient(HttpClient httpClient, String baseUri) {
        this.httpClient = httpClient;
        this.baseUri = baseUri;
    }

    public Api.TApiResponse request(PassportUid uid, UserFeatures userFeatures, ListF<String> variants) {
        return request(requestForParams(uid, userFeatures, variants));
    }

    public static Api.TApiRequest requestForParams(PassportUid uid, UserFeatures userFeatures, ListF<String> variants) {
        return Api.TApiRequest
                .newBuilder()
                .setContext(Api.TApiRequest.TContext
                        .newBuilder()
                        .setPassportUid(uid.toString())
                        .addAllFeatures(userFeatures.toParams())
                        .build())
                .addAllItems(variants)
                .build();
    }

    public Api.TApiResponse request(Api.TApiRequestOrBuilder request) {
        HttpPost post = new HttpPost(baseUri);
        post.setHeader(HttpHeaderNames.CONTENT_TYPE, "application/json");
        post.setHeader(HttpHeaderNames.ACCEPT, "application/json");

        try {
            String requestStr = JsonFormat.printer().print(request);
            if (logger.isTraceEnabled()) {
                logger.trace("rtx request: {}", requestStr);
            }
            post.setEntity(new ByteArrayEntity(requestStr.getBytes()));
        } catch (InvalidProtocolBufferException e) {
            throw ExceptionUtils.translate(e);
        }

        return ApacheHttpClientUtils.execute(post, httpClient, new Abstract200ExtendedResponseHandler<Api.TApiResponse>() {
            @Override
            protected Api.TApiResponse handle200Response(HttpResponse response) throws IOException {
                String responseStr = InputStreamSourceUtils2.wrap(response.getEntity().getContent()).readText();

                if (logger.isTraceEnabled()) {
                    logger.trace("rtx response: {}", responseStr);
                }

                try {
                    Api.TApiResponse.Builder builder = Api.TApiResponse.newBuilder();
                    JsonFormat.parser().merge(new StringReader(responseStr), builder);
                    return builder.build();
                } catch (InvalidProtocolBufferException | RuntimeException e) {
                    throw new BadRtxResponseException("Failed to parse response: " + responseStr, e);
                }
            }
        });
    }

    @Data
    public static class UserFeatures {
        public final String locale;

        ListF<Api.TApiRequest.TParam> toParams() {
            return Cf.map("locale", locale)
                    .entries()
                    .map(e -> Api.TApiRequest.TParam.newBuilder().setKey(e._1).setValue(e._2).build());
        }

        public final String toJson() {
            try {
                StringWriter sw = new StringWriter();
                JsonGenerator jg = new JsonFactory().createGenerator(sw);

                jg.writeStartObject();
                for (Api.TApiRequest.TParam param : toParams()) {
                    jg.writeFieldName(param.getKey());
                    jg.writeString(param.getValue());
                }
                jg.writeEndObject();

                jg.flush();

                return sw.toString();

            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }
    }
}
