package ru.yandex.chemodan.app.notifier.rtx;

import NRtx.NProto.Api;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.util.AppNameHolder;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.chemodan.util.test.HttpRecorderRule;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.version.SimpleAppName;

/**
 * @author tolmalev
 */
@ContextConfiguration(
        classes = {
                RtxClientContextConfiguration.class,
                RtxClientTest.Config.class,
                RtxMockConfiguration.class
        }
)
public class RtxClientTest extends AbstractTest {

    private static final Logger logger = LoggerFactory.getLogger(RtxClientTest.class);

    @Rule
    @Autowired
    public final HttpRecorderRule httpRecorderRule = null;

    @Autowired
    private RtxClient rtxClient;

    @BeforeClass
    public static void setup() {
        AppNameHolder.setIfNotPresent(new SimpleAppName("disk", "notifier"));
        AbstractTest.setup();
    }

    @Test
    public void serializeEmptyRequest() throws InvalidProtocolBufferException {
        Api.TApiRequest request = Api.TApiRequest
                .newBuilder()
                .build();

        logger.info("Request json: {}", JsonFormat.printer().print(request));
    }

    @Test
    public void serializeRealRequest() throws InvalidProtocolBufferException {
        Api.TApiRequest request = consRealRequest();

        logger.info("Request json: {}", JsonFormat.printer().print(request));
    }

    private static Api.TApiRequest consRealRequest() {
        return RtxClient.requestForParams(
                PassportUid.cons(50273844),
                new RtxClient.UserFeatures("ru"),
                Cf.list("autoupload_type1", "autoupload_type2", "autoupload_type3")
        );
    }

    @Test
    public void parseResponse() throws InvalidProtocolBufferException {
        String jsonResponse = "{\n" +
                " \"objects\": [\n" +
                "  \"task1\",\n" +
                "  \"task2\",\n" +
                "  \"task3\"\n" +
                " ],\n" +
                " \"RtxReqid\": \"{\\\"Id\\\":\\\"8a936de6f676555c\\\",\\\"RankerMeta\\\":{\\\"RankerId\\\":\\\"random\\\",\\\"Randomization\\\":\\\"Full\\\"}}\"\n" +
                "}";

        Api.TApiResponse.Builder builder = Api.TApiResponse.newBuilder();
        JsonFormat.parser().merge(jsonResponse, builder);
        builder.build();
    }

    @Test
    public void emptyRequest() {
        Api.TApiRequest request = Api.TApiRequest
                .newBuilder()
                .build();

        System.out.println(rtxClient.request(request));
    }

    @Test(expected = BadRtxResponseException.class)
    public void emptyRequestBadResponse() {
        Api.TApiRequest request = Api.TApiRequest
                .newBuilder()
                .build();

        System.out.println(rtxClient.request(request));
    }

    @Test
    public void simpleRequest() {
        ListF<String> items = Cf.list("item1", "item2", "item3");

        Api.TApiRequest request = Api.TApiRequest
                .newBuilder()
                .addAllItems(items)
                .build();

        Api.TApiResponse response = rtxClient.request(request);
        logger.info("rtx response: {}", response);

        Assert.sizeIs(3, response.getItemsList());
        for (String respItem : response.getItemsList()) {
            Assert.in(respItem, items);
        }
        for (String item : items) {
            Assert.in(item, response.getItemsList());
        }

        Assert.notEmpty(response.getRtxReqid());
        Assert.notEmpty(response.getRtxReqid().trim());
    }

    @Test
    public void realRequest() {
        Api.TApiResponse response = rtxClient.request(consRealRequest());
        logger.info("rtx response: {}", response);
    }

    @Test
    public void userFeaturesToJsonString() {
        RtxClient.UserFeatures features = new RtxClient.UserFeatures("ru");

        Assert.equals("{\"locale\":\"ru\"}", features.toJson());
    }

    public static class Config {
        @Bean
        public HttpRecorderRule httpRecorderRule() {
            return HttpRecorderRule.builder().prefix(System.getProperty("user.home") + "/arcadia/disk/notifier/src/test/resources/").build();
        }
    }
}
