package ru.yandex.chemodan.app.notifier.settings;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.NotificationGroup;
import ru.yandex.chemodan.app.notifier.notification.NotificationService;
import ru.yandex.chemodan.app.notifier.notification.NotificationServiceWidgetSettings;
import ru.yandex.chemodan.notifier.NotifierDatabasing;

/**
 * @author bursy
 */
@RequiredArgsConstructor
public class NotifierWidgetSettingsManager {
    private static final String GROUPS_INFIX = "_groups_";
    private static final String SERVICE_NAMES_PREFIX = "service_names_";

    private static final String TEXT_FIELD_NAME = "text";
    private static final RecordField<String> TEXT_FIELD = RecordField.string(TEXT_FIELD_NAME);
    private static final String NAME_FIELD_NAME = "name";
    private static final RecordField<String> NAME_FIELD = RecordField.string(NAME_FIELD_NAME);

    private final DataApiManager dataApiManager;

    public static final String SERVICES_COLLECTION = NotifierDatabasing.SERVICES_COLLECTION;

    public MapF<NotifierLanguage, String> getGroupNames(NotificationGroup group) {
        return getSingleFieldDataByLanguage(
                getOrCreateServicesDatabase(group.getService().isYaTeam),
                group.name,
                l -> getGroupsCollectionName(group.getService(), l),
                TEXT_FIELD::get);
    }

    public void saveGroupNames(String groupName, NotificationService service, MapF<NotifierLanguage, String> names) {
        saveSingleFieldDataByLanguage(
                getOrCreateServicesDatabase(service.isYaTeam),
                groupName,
                l -> getGroupsCollectionName(service, l),
                TEXT_FIELD_NAME,
                names
        );
    }

    public MapF<NotifierLanguage, String> getServiceNames(NotificationService service) {
        return getSingleFieldDataByLanguage(
                getOrCreateServicesDatabase(service.isYaTeam),
                service.name,
                this::getServiceNamesCollectionName,
                NAME_FIELD::get);
    }

    public void saveServiceNames(NotificationService service, MapF<NotifierLanguage, String> names) {
        saveSingleFieldDataByLanguage(
                getOrCreateServicesDatabase(service.isYaTeam),
                service.name,
                this::getServiceNamesCollectionName,
                NAME_FIELD_NAME,
                names
        );
    }

    public NotificationServiceWidgetSettings getServiceWidgetSettings(NotificationService service) {
        return dataApiManager.getRecords(getOrCreateServicesDatabase(service.isYaTeam).spec(), RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.eq(SERVICES_COLLECTION))
                .withRecordId(service.name))
                .singleO()
                .map(NotificationServiceWidgetSettings::fromDataRecord)
                .getOrElse(() -> NotificationServiceWidgetSettings.createDefault(service.name));
    }

    public void saveServiceWidgetSettings(NotificationService service, NotificationServiceWidgetSettings settings) {
        Database database = getOrCreateServicesDatabase(service.isYaTeam);
        RecordChange change = RecordChange.set(SERVICES_COLLECTION, service.name, settings.toData());
        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(change));
    }

    private MapF<NotifierLanguage, String> getSingleFieldDataByLanguage(Database database, String recordId,
            Function<NotifierLanguage, String> collectionNameMapper, Function<DataRecord, String> fieldExtractor)
    {
        MapF<NotifierLanguage, String> collectionNameMap =
                Cf.x(NotifierLanguage.values()).toMapMappingToValue(collectionNameMapper);

        MapF<String, String> dataMap = dataApiManager.getRecords(database.spec(), RecordsFilter.DEFAULT
                .withCollectionIdCond(CollectionIdCondition.inSet(collectionNameMap.values()))
                .withRecordId(recordId))
                .toMap(DataRecord::getCollectionId, fieldExtractor);

        return collectionNameMap.mapValuesWithKey((language, name) -> dataMap.getTs(name));
    }

    private void saveSingleFieldDataByLanguage(Database database, String recordId,
            Function<NotifierLanguage, String> collectionNameMapper, String fieldName,
            MapF<NotifierLanguage, String> dataMap)
    {
        ListF<RecordChange> changes = Cf.list();

        for (Tuple2<NotifierLanguage, String> entry : dataMap.entries()) {
            MapF<String, DataField> data = Cf.map(fieldName, DataField.string(entry.get2()));
            changes = changes.plus1(RecordChange.set(collectionNameMapper.apply(entry.get1()), recordId, data));
        }

        dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    public Database getOrCreateServicesDatabase(boolean isYaTeam) {
        return dataApiManager.getOrCreateDatabase(isYaTeam
                ? NotifierDatabasing.SERVICES_YT_DB_SPEC
                : NotifierDatabasing.SERVICES_DB_SPEC);
    }

    private String getGroupsCollectionName(NotificationService service, NotifierLanguage language) {
        return service.name + GROUPS_INFIX + language.value();
    }

    private String getServiceNamesCollectionName(NotifierLanguage language) {
        return SERVICE_NAMES_PREFIX + language.value();
    }
}
