package ru.yandex.chemodan.app.notifier.tanker;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.utils.DataApiZkRegistry;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkPathEscaping;
import ru.yandex.inside.tanker.TankerClient;
import ru.yandex.inside.tanker.model.TankerResponse;

/**
 * Handles Tanker communication and caching. Stores translations in ZooKeeper, retrieves from Tanker if necessary,
 * knows how to reload them from Tanker and update the stored ZooKeeper data.
 *
 * @author buberman
 */
public class TankerRegistry extends DataApiZkRegistry<TankerResponse> {
    private Option<TankerResponse> tankerResponse = Option.empty();

    private final TankerClient tankerClient;

    /**
     * Additional request parameters may include "status=unapproved" for testing purposes.
     */
    private final String additionalParams;
    private final String projectId;

    public TankerRegistry(ZkPath zkPath,
            TankerClient tankerClient,
            String additionalParams,
            String projectId)
    {
        super(zkPath, TankerResponse.class, r -> "data", ZkPathEscaping.asIs);

        this.tankerClient = tankerClient;
        this.additionalParams = additionalParams;
        this.projectId = projectId;
    }

    public TankerResponse getData() {
        if (tankerResponse.isPresent()) {
            return tankerResponse.get();
        } else {
            TankerResponse response = getTankerResponse();
            updateCachedResponse(response);
            return response;
        }
    }

    public void refreshCachedState() {
        TankerResponse dataFromTanker = getTankerResponse();

        updateCachedResponse(dataFromTanker);
    }

    protected TankerResponse getTankerResponse() {
        return tankerClient.retrieveAllProjectData(projectId, additionalParams);
    }

    @Override
    protected void handleNewState(CollectionF<TankerResponse> allElements) {
        if (allElements.isNotEmpty()) {
            tankerResponse = allElements.singleO();
        }
    }

    protected void updateCachedResponse(TankerResponse response) {
        put(response);
    }

}
