package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.utils.BlackboxMultiplexed;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public class ActionMetadataProcessor implements MpfsRelatedMetadataProcessor, BlackboxRelatedMetadataProcessor {
    public static final String GO_TO_LENTA = "GO_TO_LENTA";

    private static final Logger logger = LoggerFactory.getLogger(ActionMetadataProcessor.class);

    private final BlackboxMultiplexed blackbox;
    private final MpfsClient mpfsClient;
    private final int retryCount;

    public ActionMetadataProcessor(BlackboxMultiplexed blackbox, MpfsClient mpfsClient, int retryCount) {
        this.blackbox = blackbox;
        this.mpfsClient = mpfsClient;
        this.retryCount = retryCount;
    }

    @Override
    public BlackboxMultiplexed getBlackbox() {
        return blackbox;
    }

    @Override
    public MpfsClient getMpfsClient() {
        return mpfsClient;
    }

    @Override
    public int retryCount() {
        return retryCount;
    }

    @Override
    public ListF<MetadataEntityType> getProcessingTypes() {
        return Cf.list(MetadataEntityType.ACTION);
    }

    @Override
    public void processField(MetadataEntity field, NotificationRecord record, MetadataProcessorContext context) {
        if (field.getO("block-type").isPresent()) {
            processGoToLentaBlockAction(field, record, context);
        }
    }

    private void processGoToLentaBlockAction(MetadataEntity field, NotificationRecord record, MetadataProcessorContext context) {
        // By default add go_to_lenta action
        field.put("action", GO_TO_LENTA);

        LentaRecordType blockType = LentaRecordType.R.fromValue(field.get("block-type"));

        processLentaBlock(field, context);
        switch (blockType) {
            case PUBLIC_RESOURCE:
                addCommentId(field, record);
                processPublicResource(field, DiskMetadataProcessorUtils.getResourceId(record), context);
                break;
            case PUBLIC_RESOURCE_OWNED:
                addCommentId(field, record);
                processPublicOwnedResource(field, DiskMetadataProcessorUtils.getResourceId(record),
                        record.actor, context);
                break;
            case SHARED_RESOURCE:
                addCommentId(field, record);
                break;
            case PHOTO_SELECTION_BLOCK:
            case PHOTO_REMIND_BLOCK:
            case CONTENT_BLOCK:
                break;
            default:
                logger.warn("Unexpected block type: " + blockType);
        }
    }

    private void addCommentId(MetadataEntity field, NotificationRecord record) {
        Option<String> commentId = record.metadata.getEntityField("comment", "id");
        if (commentId.isPresent()) {
            field.put("comment_id", commentId.get());
        }
    }

    protected void processLentaBlock(MetadataEntity field, MetadataProcessorContext context) {
        Option<String> uid = field.getO("uid");
        if (!uid.isPresent()) {
            logger.warn("UID missing when processing lenta action metadata.");
            return;
        }

        BlackboxCorrectResponse userInfo = getUserInfo(uid.get(), context);
        field.put("login", userInfo.getLogin().getOrElse(""));
    }

    protected void processPublicResource(MetadataEntity field, String ownerAndResourceId,
            MetadataProcessorContext context)
    {
        logger.debug("Getting resource info for resource " + ownerAndResourceId);

        try {
            MpfsFileInfo fileInfo = getMpfsFileInfo(ownerAndResourceId, context);

            field.put("short_url", fileInfo.getMeta().getShortUrl().get());
            field.put("media_type", fileInfo.getMeta().getMediaType().getOrElse(""));
        } catch (RuntimeException e) {
            logger.warn(e, e);
        }
    }

    protected void processPublicOwnedResource(MetadataEntity field, String ownerAndResourceId,
            NotificationActor actor, MetadataProcessorContext context)
    {
        field.put("modify_uid", actor.getUid().toString());

        logger.debug("Getting resource info for resource " + ownerAndResourceId);

        try {
            MpfsFileInfo fileInfo = getMpfsFileInfo(ownerAndResourceId, context);

            field.put("file_id", fileInfo.getMeta().getFileId().get());
            field.put("media_type", fileInfo.getMeta().getMediaType().getOrElse(""));
        } catch (RuntimeException e) {
            logger.warn(e, e);
        }
    }
}
