package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import java.text.SimpleDateFormat;
import java.util.TimeZone;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.Years;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockMeta;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderInvitationBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderFields;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoSelectionFields;
import ru.yandex.chemodan.app.notifier.DiskNotificationManager;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.metadata.NotifierLanguage;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.ServiceAndType;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskTankerMessages;
import ru.yandex.chemodan.app.notifier.tanker.TankerManager;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.random.Random2;

import static ru.yandex.chemodan.app.notifier.worker.metadataprocessor.MetadataProcessorManager.FORMAT_DATE;

/**
 * @author buberman
 */
public class DiskMetadataProcessorManager {

    public final static String BLOCK_ID = "block-id";
    public final static String BLOCK_TYPE = "block-type";
    public final static String DESKTOP_URL_LINK_VALUE = "link";

    private final MetadataProcessorManager metadataProcessorManager;
    private final TankerManager tankerManager;
    private final BlockTitlesGenerator blockTitlesGenerator;

    public DiskMetadataProcessorManager(MetadataProcessorManager metadataProcessorManager, TankerManager tankerManager,
                                        BlockTitlesGenerator blockTitlesGenerator)
    {
        this.metadataProcessorManager = metadataProcessorManager;
        this.tankerManager = tankerManager;
        this.blockTitlesGenerator = blockTitlesGenerator;
    }

    public MetadataWrapper createLentaBlockMetadata(
            DataApiUserId uid, LentaBlockRecord block, DiskMetadataProcessorContext context)
    {
        MetadataWrapper metadata = new MetadataWrapper(Cf.hashMap());
        DiskNotificationManager.LentaBlockExData exData = context.getExtData();

        addLentaBlockActionData(metadata, block, uid);
        MetadataProcessorUtils.addActor(metadata, exData.actor);

        context.getContext().setUserId(uid);
        addMessageSpecificFields(metadata, block, context);

        metadataProcessorManager.processFields(metadata, null, context.getContext());

        addPreviewEntityData(metadata, exData.preview, exData.notificationType);
        if (exData.notificationType.equals(DiskNotifications.AUTOUPLOAD) ||
                exData.notificationType.equals(DiskNotifications.UNLIM_AUTOUPLOAD))
        {
            addYesterdayDateMetadata(metadata);
        }
        exData.lentaBlockDescription.map(b -> b.lentaBlock).forEach(f -> addFolderData(metadata, f));
        exData.filesCount.forEach(count -> addCountData(metadata, count));
        addReminderDateData(metadata, context);
        addSelectionDateData(metadata, context);
        addDesktopUrlData(metadata, exData);
        return metadata;
    }

    public String getGeoName(ListF<Integer> regionIds) {
        return blockTitlesGenerator.processSimplePlaceholder("geo:all.nominative",
                Language.RUSSIAN,
                new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, DateTime.now(), regionIds));
    }

    private void addDesktopUrlData(MetadataWrapper wrapper, DiskNotificationManager.LentaBlockExData exData) {
        exData.desktopUrl.ifPresent(url -> {
            MetadataEntity urlData = new MetadataEntity(MetadataEntityType.LINK);
            urlData.put(DESKTOP_URL_LINK_VALUE, url);
            wrapper.meta.put(MetadataEntityNames.DESKTOP_URL, urlData);
        });
    }

    private void addLentaBlockActionData(MetadataWrapper metadata, LentaBlockRecord block, DataApiUserId ownerUid) {
        MetadataEntity actionData = new MetadataEntity(MetadataEntityType.ACTION);
        actionData.put("uid", ownerUid.toString());
        actionData.put("action", ActionMetadataProcessor.GO_TO_LENTA);

        if (block.mTill.isPresent()) {
            actionData.put("mtill", Long.toString(block.mTill.get().getMillis()));
        }
        if (block.mFrom.isPresent()) {
            actionData.put("mfrom", Long.toString(block.mFrom.get().getMillis()));
        }
        actionData.put("mtime", Long.toString(block.mTime.getMillis()));

        actionData.put(BLOCK_ID, block.id);
        actionData.put(BLOCK_TYPE, block.type.value());
        switch (block.type) {
            case CONTENT_BLOCK:
                addAutouploadSpecificFields(block, ownerUid, actionData);
                break;
            case PHOTO_REMIND_BLOCK:
            case PHOTO_REMIND_HIDDEN_BLOCK:
                addReminderSpecificData(block, actionData);
                break;
            case PHOTO_SELECTION_BLOCK:
                addPhotoSelectionSpecificData(block, actionData);
                break;
            case SHARED_FOLDER_INVITE:
                addSharedFolderInviteSpecificFields(block, actionData);
                break;
            case SHARED_FOLDER:
                addSharedFolderSpecificFields(block, actionData);
                break;
            default:
                throw new IllegalArgumentException("Unsupported block type: " + block.type);
        }

        metadata.meta.put(MetadataEntityNames.ACTION, actionData);
    }

    private void addAutouploadSpecificFields(LentaBlockRecord block, DataApiUserId ownerUid, MetadataEntity actionData) {
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(ownerUid, block);
        actionData.put(ContentBlockFields.FOLDER_ID.name, meta.folderId);
        actionData.put(ContentBlockFields.MEDIA_TYPE.name, meta.mediaType);
        actionData.put(ContentBlockFields.MODIFIER_UID.name, meta.modifierUid);
        actionData.put(ContentBlockFields.GROUP_KEY.name, block.groupKey);
        actionData.put(ContentBlockFields.ORDER.name, Long.toString(block.order));
        meta.area.ifPresent(n -> actionData.put(ContentBlockFields.AREA.name, n));

    }

    private void addReminderSpecificData(LentaBlockRecord block, MetadataEntity actionData) {
        actionData.put(
                PhotoReminderFields.YEARS_AGO.name,
                Long.toString(PhotoReminderFields.YEARS_AGO.get(block)));
        actionData.put(
                PhotoReminderFields.INTERVAL_START.name,
                Long.toString(PhotoReminderFields.INTERVAL_START.get(block).getMillis()));
        actionData.put(
                PhotoReminderFields.INTERVAL_END.name,
                Long.toString(PhotoReminderFields.INTERVAL_END.get(block).getMillis()));
        actionData.put(PhotoReminderFields.RESOURCE_IDS.name,
                PhotoReminderFields.RESOURCE_IDS.get(block).mkString(","));
        actionData.put(PhotoReminderFields.NOTIFICATION_TYPE.name,
                PhotoReminderFields.NOTIFICATION_TYPE.getO(block).map(StringEnum::value).getOrElse(""));
    }

    private void addPhotoSelectionSpecificData(LentaBlockRecord block, MetadataEntity actionData) {
        actionData.put(
                PhotoSelectionFields.INTERVAL_START.name,
                Long.toString(PhotoSelectionFields.INTERVAL_START.get(block).getMillis()));
        actionData.put(
                PhotoSelectionFields.INTERVAL_END.name,
                Long.toString(PhotoSelectionFields.INTERVAL_END.get(block).getMillis()));

        PhotoSelectionFields.PHOTOSLICE_DATE.getO(block).forEach(date ->
                actionData.put(PhotoSelectionFields.PHOTOSLICE_DATE.name, Long.toString(date.getMillis())));

        actionData.put(PhotoSelectionFields.ENABLED_PLATFORMS.name,
                PhotoSelectionFields.ENABLED_PLATFORMS.getO(block).getOrElse(""));

        actionData.put(PhotoSelectionFields.SUBTYPE.name,
                PhotoSelectionFields.SUBTYPE.getO(block).getOrElse(""));
        actionData.put(PhotoSelectionFields.ICON_TYPE.name,
                PhotoSelectionFields.ICON_TYPE.getO(block).getOrElse(""));

        PhotoSelectionFields.TITLE.entries(block).forEach(actionData::put);
    }

    private void addSharedFolderInviteSpecificFields(LentaBlockRecord block, MetadataEntity actionData) {
        actionData.put(SharedFolderInvitationBlockFields.GROUP_ID.name,
                SharedFolderInvitationBlockFields.GROUP_ID.get(block));
        actionData.put(SharedFolderInvitationBlockFields.INVITE_HASH.name,
                SharedFolderInvitationBlockFields.INVITE_HASH.get(block));
        actionData.put(SharedFolderInvitationBlockFields.OWNER_UID.name,
                SharedFolderInvitationBlockFields.OWNER_UID.get(block));
    }

    private void addSharedFolderSpecificFields(LentaBlockRecord block, MetadataEntity actionData) {
        actionData.put(SharedFolderBlockFields.GROUP_ID.name, SharedFolderBlockFields.GROUP_ID.get(block));
        actionData.put(SharedFolderBlockFields.OWNER_UID.name, SharedFolderBlockFields.OWNER_UID.get(block));
    }

    private void addCountData(MetadataWrapper metadata, long count) {
        MetadataEntity countData = new MetadataEntity(MetadataEntityType.TEXT);
        countData.put("text", Long.toString(count));
        metadata.meta.put(MetadataEntityNames.COUNT, countData);

        metadata.put(MetadataEntityNames.ACTION, "files_count", Long.toString(count));
    }

    private void addSelectionDateData(MetadataWrapper metadata, DiskMetadataProcessorContext context) {
        DiskNotificationManager.LentaBlockExData exData = context.getExtData();
        if (!exData.coolLentaReminderDate.isPresent()) {
            return;
        }
        DiskNotificationManager.CoolLentaReminderDate reminderDate = exData.coolLentaReminderDate.get();
        DateTime start = reminderDate.intervalStart;
        IntervalType intervalType = reminderDate.intervalType;

        TitleGenerationContext titleContext = new TitleGenerationContext(Random2.R, intervalType, start,
                exData.geoRegionIds, exData.themeTerm);

        if (reminderDate.generationTime.isPresent()) {
            int nYearsAgoCount = Years.yearsBetween(start.toLocalDate(), reminderDate.generationTime.get().toLocalDate()).getYears();
            titleContext = titleContext.withAttribute(CoolLentaReminder.YEARS_ATTRIBUTE_NAME, nYearsAgoCount);
        }

        TitleGenerationContext finalTitleContext = titleContext;

        context.getMessagesPlaceholders().filter(blockTitlesGenerator::isSupportedPlaceholder).forEach(placeholder -> {
            MapF<NotifierLanguage, String> localizedValues = Cf.hashMap();

            NotifierLanguage.R.valuesList().forEach(lang -> {
                Option<Language> supportedLanguage = Language.R.valueOfO(lang.toString());
                if (!supportedLanguage.isPresent() || !BlockTitlesGenerator.SUPPORTED_LANGUAGES.containsTs(supportedLanguage.get())) {
                    return;
                }

                String text = blockTitlesGenerator.processSimplePlaceholder(placeholder, supportedLanguage.get(), finalTitleContext);

                localizedValues.put(lang, text);
            });

            MetadataEntity placeholderMetadata = new MetadataEntity(MetadataEntityType.TEXT);

            int localizationsCount = localizedValues.values().unique().size();
            if (localizationsCount == 1) {
                placeholderMetadata.put("text", localizedValues.values().toList().first());
            } else if (localizationsCount > 1) {
                localizedValues.entries().forEach(t2 -> placeholderMetadata.put(MetadataEntity.translatedTextField(t2._1), t2._2));
            }

            metadata.meta.put(placeholder, placeholderMetadata);
        });
    }

    private void addReminderDateData(MetadataWrapper metadata, DiskMetadataProcessorContext context) {
        if (!context.getExtData().reminderDate.isPresent()) {
            return;
        }

        DiskNotificationManager.PhotoReminderDate date = context.getExtData().reminderDate.get();
        Lazy<Option<Integer>> year = date.getYear();
        Lazy<Option<Integer>> month = date.getMonth();

        MetadataEntity countData = new MetadataEntity(MetadataEntityType.TEXT);
        countData.put("text", Integer.toString(date.yearsAgo));
        metadata.meta.put(MetadataEntityNames.YEARS, countData);

        if(date.getPeriodStart().get().isPresent()) {
            Instant periodStart = date.getPeriodStart().get().get();
            MetadataEntity periodStartData = new MetadataEntity(MetadataEntityType.TEXT);

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
            simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

            periodStartData.put("text", simpleDateFormat.format(periodStart.toDate()));
            metadata.meta.put(MetadataEntityNames.PERIOD_START_TEXT, periodStartData);
        }

        if (context.getMessagesPlaceholders().containsTs(MetadataEntityNames.YEAR) && year.get().isPresent()) {
            MetadataEntity yearData = new MetadataEntity(MetadataEntityType.TEXT);
            yearData.put("text", Integer.toString(year.get().get()));
            metadata.meta.put(MetadataEntityNames.YEAR, yearData);
        }

        if (context.getMessagesPlaceholders().containsTs(MetadataEntityNames.MONTH) && month.get().isPresent()) {
            MapF<String, TankerTranslation> translations = tankerManager.getAllTranslations(
                    DiskTankerMessages.MONTHS_LOCATIVE);

            MetadataEntity monthData = new MetadataEntity(MetadataEntityType.TEXT);

            NotifierLanguage.R.valuesList().forEach(lang ->
                    translations.getO(lang.value()).forEach(text ->
                            monthData.put(MetadataEntity.translatedTextField(lang),
                                    text.form.get().text.split("\\|")[month.get().get() - 1])));

            metadata.meta.put(MetadataEntityNames.MONTH, monthData);
        }
    }

    private void addPreviewEntityData(MetadataWrapper metadata, Option<DiskNotificationManager.PreviewInfo> preview, ServiceAndType type) {
        MetadataEntity entityData = new MetadataEntity(MetadataEntityType.RESOURCE);
        if (type.equals(DiskNotifications.SHARED_FOLDER_INVITE_ACCEPTED)
                || type.equals(DiskNotifications.SHARED_FOLDER_INVITE_RECEIVED))
        {
            entityData.put("resource_type", "dir");
        } else if (preview.isPresent()) {
            entityData.put(MetadataEntityNames.PREVIEW_URL, preview.get().url);
            entityData.put(MetadataEntityNames.PREVIEW_FILE_PATH, preview.get().filePath);
            entityData.put(MetadataEntityNames.PREVIEW_FILE_RESOURCE_ID, preview.get().resourceId);
        }
        metadata.meta.put(MetadataEntityNames.ENTITY, entityData);
    }

    private void addYesterdayDateMetadata(MetadataWrapper metadata) {
        MetadataEntity dateMetadata = new MetadataEntity(MetadataEntityType.DATE);
        long millis = DateTime.now().minusDays(1).getMillis();
        dateMetadata.put("ms", Long.toString(millis));
        dateMetadata.put("text", new DateTime(millis).toString(FORMAT_DATE));

        metadata.meta.put(MetadataEntityNames.DATE, dateMetadata);
    }

    private void addActor(MetadataWrapper metadata, NotificationActor actor) {
        if (actor.isUser()) {
            MetadataEntity entity = new MetadataEntity(MetadataEntityType.USER);
            entity.put("uid", actor.getUid().toString());
            metadata.meta.put(MetadataEntityNames.ACTOR, entity);
        } else {
            MetadataEntity actorData = new MetadataEntity(MetadataEntityType.YA_DISK);
            // TODO: do we need localized messages for YaDisk here?
            metadata.meta.put(MetadataEntityNames.ACTOR, actorData);
        }
    }

    private void addFolderData(MetadataWrapper metadataWrapper, MpfsLentaBlockItemDescription folderBlock) {
        MetadataEntity folderEntity = new MetadataEntity(MetadataEntityType.RESOURCE);
        folderEntity.put("id", folderBlock.resourceId);
        folderEntity.put("text", folderBlock.name);
        folderBlock.url.forEach(url -> folderEntity.put("short_url", url));

        metadataWrapper.meta.put(MetadataEntityNames.FOLDER, folderEntity);
    }

    private void addMessageSpecificFields(
            MetadataWrapper metadata, LentaBlockRecord block, DiskMetadataProcessorContext context)
    {
        if (context.getMessagesPlaceholders().containsTs(MetadataEntityNames.TOTAL_PHOTOS)) {
            metadata.meta.put(MetadataEntityNames.TOTAL_PHOTOS, new MetadataEntity(MetadataEntityType.TOTAL_PHOTOS));
        }
    }

}
