package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import java.util.Arrays;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Years;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockMeta;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.SharedFolderInvitationBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GeoNamesSource;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoReminderFields;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.BlockTexts;
import ru.yandex.chemodan.app.notifier.DiskNotificationManager;
import ru.yandex.chemodan.app.notifier.inflector.InflectorResponse;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskNotifications;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskTankerMessages;
import ru.yandex.chemodan.app.notifier.tanker.TankerManager;
import ru.yandex.chemodan.app.notifier.utils.BlackboxMultiplexed;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.inside.tanker.model.FormWithCases;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.mime.detect.MediaType;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * @author buberman
 */
@RunWith(MockitoJUnitRunner.Silent.class)
public class DiskMetadataProcessorManagerTest {

    private static DiskMetadataProcessorManager diskMetadataProcessorManager;
    private static MetadataProcessorManager metadataProcessorManager;

    private static MpfsClient mpfsClient = mock(MpfsClientImpl.class);

    private static DiskSearchClient diskSearchClient = mock(DiskSearchClient.class);

    private static TankerManager tankerManager = mock(TankerManager.class);
    private static BlockTitlesGenerator blockTitlesGenerator = new BlockTitlesGenerator(false, GeoNamesSource.EMPTY);

    @BeforeClass
    public static void prepareProcessor() {
        InflectorResponse inflectorResponse = new InflectorResponse();
        inflectorResponse.forms = Cf.map("им", "Display name", "дат", "Display dative");

        when(diskSearchClient.findPhotosliceDocumentsCount(any())).thenReturn(100500L);

        // TODO: test new implementation of UserMetadataProcessor
        BlackboxMultiplexed blackbox = MetadataProcessorTestUtils.createBlackboxMock();
        metadataProcessorManager = new MetadataProcessorManager(Arrays.asList
                (
                        new CommentMetadataProcessor(mpfsClient, new TextMetadataProcessor(-1, -1), 0),
                        new ResourceMetadataProcessor(mpfsClient, 0),
                        new UserMetadataProcessor(blackbox, 0),
                        new TextMetadataProcessor(-1, -1),
                        new ActionMetadataProcessor(blackbox, mpfsClient, 0),
                        new TotalPhotosMetadataProcessor(diskSearchClient),
                        new DateMetadataProcessor()
                ));

        TankerTranslation translation = new TankerTranslation();
        translation.form = Option.of(new FormWithCases("январе|феврале|...", Cf.map()));

        when(tankerManager.getAllTranslations(eq(DiskTankerMessages.MONTHS_LOCATIVE)))
                .thenReturn(Cf.map("ru", translation));

        diskMetadataProcessorManager = new DiskMetadataProcessorManager(metadataProcessorManager, tankerManager, blockTitlesGenerator);
    }


    @Test
    public void createAutouploadLentaBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        LentaBlockRecord lentaBlock = createLentaContentBlock(ContentBlockAction.AUTOSAVE);
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, lentaBlock);

        DiskNotificationManager.PreviewInfo preview = new DiskNotificationManager.PreviewInfo(
                "http://preview.ru/url",
                "file_path",
                "resource_id"
        );

        DiskNotificationManager.LentaBlockExData data = DiskNotificationManager.LentaBlockExData.createContentData(
                Option.of((long) meta.filesCount), Option.of(preview), DiskNotifications.AUTOUPLOAD,
                NotificationActor.YA_DISK, Option.empty());

        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock,
                new DiskMetadataProcessorContext(new MetadataProcessorContext(), data, Cf.set("total_photos")));

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("entity").get());
        Assert.equals(preview.url, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL).get());
        Assert.equals(preview.filePath, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_PATH).get());
        Assert.equals(preview.resourceId, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_RESOURCE_ID).get());

        Assert.equals(MetadataEntityType.TEXT, metadata.getEntityType("count").get());
        Assert.equals(Integer.toString(meta.filesCount), metadata.getEntityField("count", "text").get());

        Assert.some(MetadataEntityType.TOTAL_PHOTOS, metadata.getEntityType("total_photos"));
        Assert.some("100500", metadata.getEntityField("total_photos", "text"));

        Assert.equals(MetadataEntityType.YA_DISK, metadata.getEntityType("actor").get());

        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());
        Assert.equals(LentaRecordType.CONTENT_BLOCK.value(), metadata.getEntityField("action", "block-type").get());
        Assert.equals(meta.folderId, metadata.getEntityField("action", "folder_id").get());
        Assert.equals(meta.mediaType, metadata.getEntityField("action", "media_type").get());
        Assert.equals(meta.modifierUid, metadata.getEntityField("action", "modifier_uid").get());
        Assert.equals("1477047600000", metadata.getEntityField("action", "mtill").get());
        Assert.equals("1477011600000", metadata.getEntityField("action", "mfrom").get());
        Assert.equals("1477047600000", metadata.getEntityField("action", "mtime").get());
    }

    @Test
    public void createPhotoReminderLentaBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        LentaBlockRecord lentaBlock = createPhotoReminderLentaBlock();
        DiskNotificationManager.PreviewInfo preview = new DiskNotificationManager.PreviewInfo(
                "http://preview.ru/url",
                "file_path",
                "resource_id"
        );

        DiskNotificationManager.LentaBlockExData data =
                DiskNotificationManager.LentaBlockExData.createPhotoReminderData(Option.of(preview),
                        DiskNotifications.PHOTO_REMINDER,
                        new DiskNotificationManager.PhotoReminderDate(1, 2017, 1), NotificationActor.YA_DISK);

        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock,
                new DiskMetadataProcessorContext(new MetadataProcessorContext(), data, Cf.set("month", "year")));

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("entity").get());
        Assert.equals(preview.url, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL).get());
        Assert.equals(preview.filePath, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_PATH).get());
        Assert.equals(preview.resourceId, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_RESOURCE_ID).get());

        Assert.equals(MetadataEntityType.YA_DISK, metadata.getEntityType("actor").get());

        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());
        Assert.equals(LentaRecordType.PHOTO_REMIND_BLOCK.value(), metadata.getEntityField("action", "block-type").get());

        Assert.equals("1", metadata.getEntityField("action", "years_ago").get());
        Assert.equals("resource_id1,resource_id2", metadata.getEntityField("action", "resource_ids").get());
        Assert.equals("1468580400000", metadata.getEntityField("action", "interval_start").get());
        Assert.equals("1477220400000", metadata.getEntityField("action", "interval_end").get());

        Assert.equals("1477220400000", metadata.getEntityField("action", "mtill").get());
        Assert.equals("1477184400000", metadata.getEntityField("action", "mfrom").get());
        Assert.equals("1477220400000", metadata.getEntityField("action", "mtime").get());
        Assert.equals("1", metadata.getEntityField("years", "text").get());

        Assert.some("2017", metadata.getEntityField("year", "text"));

        Assert.some("январе", metadata.getEntityField("month", "text@ru"));
        Assert.none(metadata.getEntityField("month", "text"));
    }

    @Test
    public void createCoolLentaRemindBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        DiskNotificationManager.PreviewInfo preview = new DiskNotificationManager.PreviewInfo(
                "http://preview.ru/url",
                "file_path",
                "resource_id"
        );

        DateTime intervalStart = DateTime.parse("2018-09-01T00:00:00.00+0200");
        DateTime intervalEnd = DateTime.parse("2018-10-01T00:00:00.00+0200");

        LentaBlockRecord lentaBlock = createCoolLentRemindBlock(
                intervalStart,
                intervalEnd,
                new DynamicLocalizedString(Cf.map(
                        Language.RUSSIAN, "Подборка. Сентябрь 2018",
                        Language.ENGLISH, "Selection of photos. September 2018",
                        Language.UKRAINIAN, "Добiрка. Вересень 2018"
                ))
        );

        DiskNotificationManager.LentaBlockExData data =
                DiskNotificationManager.LentaBlockExData.coolLentaRemindData(Option.of(preview),
                        DiskNotifications.PHOTO_SELECTION_COOL_LENTA,
                        Cf.list("ios", "android"),
                        new DiskNotificationManager.CoolLentaReminderDate(intervalStart, intervalEnd, Option.of(intervalStart.plus(Years.years(3)))), Cf.list(),
                        Option.empty(), Option.empty());

        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock,
                new DiskMetadataProcessorContext(new MetadataProcessorContext(), data, Cf.set(
                        "date.start:yyyy",
                        "date.end:yyyy",
                        "month.start:nom",
                        "season.start:nom",
                        "attributes:n.years"
                )));

        Assert.some("2018", metadata.getEntityField("date.start:yyyy", "text"));
        Assert.some("2018", metadata.getEntityField("date.end:yyyy", "text"));

        Assert.some("3", metadata.getEntityField("attributes:n.years", "text"));

        Assert.some("сентябрь", metadata.getEntityField("month.start:nom", "text@ru"));
        Assert.some("September", metadata.getEntityField("month.start:nom", "text@en"));
        Assert.some("вересень", metadata.getEntityField("month.start:nom", "text@uk"));

        Assert.some("осень", metadata.getEntityField("season.start:nom", "text@ru"));
        Assert.some("fall", metadata.getEntityField("season.start:nom", "text@en"));
        Assert.some("осінь", metadata.getEntityField("season.start:nom", "text@uk"));
    }

    @Test
    public void createSharedFolderLentaBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        LentaBlockRecord lentaBlock = createLentaContentBlock(ContentBlockAction.ADDITION);
        ContentBlockMeta meta = ContentBlockMeta.fromBlock(uid, lentaBlock);
        DiskNotificationManager.PreviewInfo preview = new DiskNotificationManager.PreviewInfo(
                "http://preview.ru/url",
                "file_path",
                "resource_id"
        );

        MpfsLentaBlockFullDescription descr = createMpfsLentaBlockFullDescription();
        DiskNotificationManager.LentaBlockExData data = DiskNotificationManager.LentaBlockExData.createContentData(
                Option.empty(), Option.of(preview), DiskNotifications.SHARED_FOLDER_FILE_CREATE,
                NotificationActor.consFromUid(123), Option.of(descr));
        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock, context(data));

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("entity").get());
        Assert.equals(preview.url, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_URL).get());
        Assert.equals(preview.filePath, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_PATH).get());
        Assert.equals(preview.resourceId, metadata.getEntityField("entity", MetadataEntityNames.PREVIEW_FILE_RESOURCE_ID).get());

        Assert.equals(MetadataEntityType.USER, metadata.getEntityType("actor").get());
        Assert.some("123", metadata.getEntityField("actor", "uid"));
        Assert.some("Public N.", metadata.getEntityField("actor", "text"));

        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());
        Assert.equals(LentaRecordType.CONTENT_BLOCK.value(), metadata.getEntityField("action", "block-type").get());

        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());
        Assert.equals(LentaRecordType.CONTENT_BLOCK.value(), metadata.getEntityField("action", "block-type").get());
        Assert.equals(meta.folderId, metadata.getEntityField("action", "folder_id").get());
        Assert.equals(meta.mediaType, metadata.getEntityField("action", "media_type").get());
        Assert.equals(meta.modifierUid, metadata.getEntityField("action", "modifier_uid").get());
        Assert.equals("1477047600000", metadata.getEntityField("action", "mtill").get());
        Assert.equals("1477011600000", metadata.getEntityField("action", "mfrom").get());
        Assert.equals("1477047600000", metadata.getEntityField("action", "mtime").get());

        Assert.some(descr.lentaBlock.resourceId, metadata.getEntityField("folder", "id"));
        Assert.some(descr.lentaBlock.name, metadata.getEntityField("folder", "text"));
        Assert.some(descr.lentaBlock.url.get(), metadata.getEntityField("folder", "short_url"));
    }

    @Test
    public void createSharedFolderInviteReceivedLentaBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        String inviteHash = "3fccae609b19603c5022dd9605b3eeb1";
        String groupId = "56a4ecba94f386ff2fb6df32dd90e5c9";
        String ownerUid = "321";
        String folderName = "my_shared_folder";
        String folderResourceId = ownerUid + ":/disk/" + folderName;
        LentaBlockRecord lentaBlock = createSharedFolderInviteReceivedLentaBlock(inviteHash, groupId, ownerUid);

        DiskNotificationManager.LentaBlockExData data =
                DiskNotificationManager.LentaBlockExData.createSharedFolderData(
                        DiskNotifications.SHARED_FOLDER_INVITE_RECEIVED,
                        NotificationActor.consFromUid(Long.parseLong(ownerUid)),
                        Option.of(createSharedFolderInviteLentaBlockDescription(folderResourceId, folderName)));

        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock, context(data));

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("entity").get());
        Assert.equals("dir", metadata.getEntityField("entity", "resource_type").get());

        Assert.equals(MetadataEntityType.USER, metadata.getEntityType("actor").get());
        Assert.equals(ownerUid, metadata.getEntityField("actor", "uid").get());
        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());

        Assert.equals(LentaRecordType.SHARED_FOLDER_INVITE.value(),
                metadata.getEntityField("action", "block-type").get());

        Assert.equals(inviteHash, metadata.getEntityField("action", "invite_hash").get());
        Assert.equals(groupId, metadata.getEntityField("action", "group_id").get());
        Assert.equals(ownerUid, metadata.getEntityField("action", "owner_uid").get());

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("folder").get());
        Assert.equals(folderName, metadata.getEntityField("folder", "text").get());
        Assert.equals(folderResourceId, metadata.getEntityField("folder", "id").get());

        Assert.equals("1505233860000", metadata.getEntityField("action", "mtill").get());
        Assert.equals("1505233860000", metadata.getEntityField("action", "mfrom").get());
        Assert.equals("1505233860000", metadata.getEntityField("action", "mtime").get());
    }

    @Test
    public void createSharedFolderInviteAcceptedLentaBlockMetadata() {
        DataApiUserId uid = DataApiUserId.parse("123");
        String groupId = "56a4ecba94f386ff2fb6df32dd90e5c9";
        String ownerUid = "321";
        String folderName = "my_shared_folder";
        String folderResourceId = ownerUid + ":/disk/" + folderName;
        LentaBlockRecord lentaBlock = createSharedFolderInviteAcceptedLentaBlock(groupId, ownerUid);

        DiskNotificationManager.LentaBlockExData data =
                DiskNotificationManager.LentaBlockExData.createSharedFolderData(
                        DiskNotifications.SHARED_FOLDER_INVITE_ACCEPTED, NotificationActor.YA_DISK,
                        Option.of(createSharedFolderInviteLentaBlockDescription(folderResourceId, folderName)));

        MetadataWrapper metadata = diskMetadataProcessorManager.createLentaBlockMetadata(uid, lentaBlock, context(data));

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("entity").get());
        Assert.equals("dir", metadata.getEntityField("entity", "resource_type").get());

        Assert.equals(MetadataEntityType.YA_DISK, metadata.getEntityType("actor").get());
        Assert.equals(MetadataEntityType.ACTION, metadata.getEntityType("action").get());
        Assert.equals(ActionMetadataProcessor.GO_TO_LENTA, metadata.getEntityField("action", "action").get());

        Assert.equals(LentaRecordType.SHARED_FOLDER.value(), metadata.getEntityField("action", "block-type").get());

        Assert.equals(groupId, metadata.getEntityField("action", "group_id").get());
        Assert.equals(ownerUid, metadata.getEntityField("action", "owner_uid").get());

        Assert.equals(MetadataEntityType.RESOURCE, metadata.getEntityType("folder").get());
        Assert.equals(folderName, metadata.getEntityField("folder", "text").get());
        Assert.equals(folderResourceId, metadata.getEntityField("folder", "id").get());

        Assert.equals("1505233740000", metadata.getEntityField("action", "mtime").get());
    }

    private MpfsLentaBlockFullDescription createMpfsLentaBlockFullDescription() {
        return new MpfsLentaBlockFullDescription(
                10,
                new MpfsLentaBlockItemDescription(
                        "id", "file-id", "resource-id",
                        Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                        "/disk/some/path/folder", "folder", "type",
                        Option.of("https://yadi.sk/33kLnp"), Option.empty()),
                Cf.list());
    }

    private LentaBlockRecord createLentaContentBlock(ContentBlockAction action) {
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                ContentBlockFields.MEDIA_TYPE.toData(MediaType.IMAGE.getName()),
                ContentBlockFields.FOLDER_ID.toData("folderId"),
                ContentBlockFields.MODIFIER_UID.toData("123"),
                ContentBlockFields.FILES_COUNT.toData(10),
                ContentBlockFields.ACTION.toData(action)));

        Instant now = new DateTime(2016, 10, 21, 14, 0, 0, MoscowTime.TZ).toInstant();
        return new LentaBlockRecord(
                "id", 1, LentaRecordType.CONTENT_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createPhotoReminderLentaBlock() {
        Instant now = new DateTime(2016, 10, 23, 14, 0, 0, MoscowTime.TZ).toInstant();
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                PhotoReminderFields.YEARS_AGO.toData(1),
                PhotoReminderFields.RESOURCE_IDS.toData(Cf.list("resource_id1", "resource_id2")),
                PhotoReminderFields.INTERVAL_START.toData(now.minus(Duration.standardDays(100))),
                PhotoReminderFields.INTERVAL_END.toData(now)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.PHOTO_REMIND_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createSharedFolderInviteReceivedLentaBlock(String inviteHash, String groupId, String ownerUid) {
        Instant now = new DateTime(2017, 9, 12, 19, 31, 0, MoscowTime.TZ).toInstant();
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                SharedFolderInvitationBlockFields.INVITE_HASH.toData(inviteHash),
                SharedFolderInvitationBlockFields.GROUP_ID.toData(groupId),
                SharedFolderInvitationBlockFields.OWNER_UID.toData(ownerUid)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.SHARED_FOLDER_INVITE,
                "groupkey",
                Option.of(now),
                now,
                Option.of(now),
                specific);
    }

    private LentaBlockRecord createSharedFolderInviteAcceptedLentaBlock(String groupId, String ownerUid) {
        Instant now = new DateTime(2017, 9, 12, 19, 29, 0, MoscowTime.TZ).toInstant();
        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                SharedFolderBlockFields.GROUP_ID.toData(groupId),
                SharedFolderBlockFields.OWNER_UID.toData(ownerUid)));

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.SHARED_FOLDER,
                "groupkey",
                Option.empty(),
                now,
                Option.empty(),
                specific);
    }

    private LentaBlockRecord createCoolLentRemindBlock(DateTime from, DateTime to, LocalizedString title) {
        Instant now = new DateTime(2017, 9, 12, 19, 29, 0, MoscowTime.TZ).toInstant();

        MinimalCoolLentaBlock block = new MinimalCoolLentaBlock(
                "id",
                "default_" + TimeIntervalUtils.getMinimalInterval(from, to),
                Cf.list("50273844:123125415135").map(MpfsResourceId::parse),
                MpfsResourceId.parse("50273844:123125415135"),
                from.getZone().getID(),
                Option.empty(),
                from.toInstant(),
                to.toInstant(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                false
        );
        MapF<String, DataField> specific = CoolLentaReminder.getSpecificLentaBlockFields(CoolLentaReminder.BlockGenerationType.USUAL, block,
                new BlockTexts(new TitleGenerationContext(Random2.R, IntervalType.ONE_DAY, DateTime.now()),
                        title, title, title, title, Cf.list()), Cf.list("ios", "android"), DateTime.now(), Option.empty());

        return new LentaBlockRecord(
                "id", 1, LentaRecordType.PHOTO_SELECTION_BLOCK,
                "groupkey",
                Option.of(now.minus(Duration.standardHours(10))),
                now,
                Option.of(now),
                specific);
    }

    private MpfsLentaBlockFullDescription createSharedFolderInviteLentaBlockDescription(String id, String name) {
        return new MpfsLentaBlockFullDescription(
                1,
                new MpfsLentaBlockItemDescription(id, id, id,
                        Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                        "/disk/some/path/folder", name, "dir", Option.empty(), Option.empty()),
                Cf.list());
    }

    private static DiskMetadataProcessorContext context(DiskNotificationManager.LentaBlockExData data) {
        return new DiskMetadataProcessorContext(new MetadataProcessorContext(), data, Cf.set());
    }

}
