package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import java.util.Arrays;

import org.joda.time.Instant;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.notifier.NotificationTestUtils;
import ru.yandex.chemodan.app.notifier.inflector.InflectorClient;
import ru.yandex.chemodan.app.notifier.inflector.InflectorResponse;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.notification.NotificationActor;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.utils.BlackboxMultiplexed;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.misc.test.Assert;

import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.misc.test.Assert.assertTrue;

/**
 * @author buberman
 */
@RunWith(MockitoJUnitRunner.class)
public class MetadataProcessorManagerTest {
    public static final long RESOURCE_OWNER_LONG_ID = 123456789l;
    public static final String RESOURCE_OWNER_ID = Long.toString(RESOURCE_OWNER_LONG_ID);
    public static final String RESOURCE_ID = RESOURCE_OWNER_ID + ":QWERTYUIOP12";
    public static final String COMMENT_ID = "456456456";

    private static MetadataProcessorManager metadataProcessorManager;

    private static MpfsClient mpfsClient = mock(MpfsClientImpl.class);

    @BeforeClass
    public static void prepareProcessor() {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder()
                .resourceId(MpfsResourceId.parse("uid:fileid"))
                .source("disk")
                .fileId("fileid")
                .publicHash("publicHash1234567")
                .shortUrl("https://yadi.sk/shorturl")
                .shortNamed("https://yadi.sk/shortnamed")
                .build();

        MpfsFileInfo fileInfo = new MpfsFileInfo("Filename", "file", meta);

        when(mpfsClient.getFileInfoByFileId(any(), any())).thenReturn(fileInfo);

        MpfsFileMetaDto meta2 = MpfsFileMetaDto.builder()
                .resourceId(MpfsResourceId.parse("uid:fileid"))
                .source("disk")
                .fileId("fileid")
                .preview("https://preview.ru")
                .build();

        MpfsFileInfo publicFileInfo = new MpfsFileInfo("Filename", "file", meta2);

        when(mpfsClient.getFileInfoOByPrivateHash(any())).thenReturn(Option.of(publicFileInfo));

        InflectorClient inflectorClient = mock(InflectorClient.class);
        InflectorResponse inflectorResponse = new InflectorResponse();
        inflectorResponse.forms = Cf.map("им", "Display name", "дат", "Display dative");

        when(inflectorClient.inflectFio(eq("Display name"), any(), any()))
                .thenReturn(inflectorResponse);

        // TODO: test new implementation of UserMetadataProcessor
        BlackboxMultiplexed blackbox = MetadataProcessorTestUtils.createBlackboxMock();
        metadataProcessorManager = new MetadataProcessorManager(Arrays.asList
                (
                        new CommentMetadataProcessor(mpfsClient, new TextMetadataProcessor(-1, -1), 0),
                        new ResourceMetadataProcessor(mpfsClient, 0),
                        new UserMetadataProcessor(blackbox, 0),
                        new TextMetadataProcessor(-1, -1),
                        new ActionMetadataProcessor(blackbox, mpfsClient, 0),
                        new DateMetadataProcessor()
                ));
    }

    @Test
    public void processKnownFields() {
        NotificationActor notificationActor = NotificationActor.consFromUid(9999);

        MetadataWrapper metadataWrapper = createMetaData(notificationActor);

        NotificationRecord record = new NotificationRecord(
                "646464",
                notificationActor,
                Option.of(RESOURCE_ID),
                "ZXCVBNM007",
                NotificationTestUtils.NOTIFICATION_TYPE_FOR_TESTS,
                metadataWrapper,
                Instant.now()
        );

        MetadataWrapper newMeta = metadataProcessorManager
                .createNotificationMetadata(record, Option.of(5L));
        assertNotNull(newMeta.meta);

        MapF processedMetadata = newMeta.meta;

        assertTrue(processedMetadata.containsKeyTs("comment"));
        assertTrue(processedMetadata.getTs("comment") instanceof MetadataEntity);

        MetadataEntity commentMetadata = (MetadataEntity) processedMetadata.getTs("comment");
        Assert.equals(MetadataEntityType.COMMENT, commentMetadata.getEntityType());
        Assert.equals(COMMENT_ID, commentMetadata.get("id"));
        Assert.equals(RESOURCE_ID, commentMetadata.get("resource_id"));
        Assert.equals("https://yadi.sk/shorturl?cid=456456456", commentMetadata.get("link"));
        Assert.equals("Вася дурак!", commentMetadata.get("text"));

        assertTrue(processedMetadata.containsKeyTs("actor"));
        assertTrue(processedMetadata.getTs("actor") instanceof MetadataEntity);

        MetadataEntity actorMetadata = (MetadataEntity) processedMetadata.getTs("actor");
        Assert.equals(MetadataEntityType.USER, actorMetadata.getEntityType());
        Assert.equals("Public N.", actorMetadata.get("text"));

        assertTrue(processedMetadata.containsKeyTs("entity"));
        assertTrue(processedMetadata.getTs("entity") instanceof MetadataEntity);

        MetadataEntity entityMetadata = (MetadataEntity) processedMetadata.getTs("entity");
        Assert.equals(MetadataEntityType.RESOURCE, entityMetadata.getEntityType());
        Assert.equals(RESOURCE_ID, entityMetadata.get("id"));
        Assert.equals("https://yadi.sk/shorturl", entityMetadata.get("short_url"));
        Assert.equals("Filename", entityMetadata.get("text"));
        Assert.equals("file", entityMetadata.get("resource_type"));
        Assert.equals(null, entityMetadata.get(MetadataEntityNames.PREVIEW_URL));

        assertTrue(processedMetadata.containsKeyTs("count"));
        assertTrue(processedMetadata.getTs("count") instanceof MetadataEntity);

        MetadataEntity textMetadata = (MetadataEntity) processedMetadata.getTs("count");
        Assert.equals(MetadataEntityType.TEXT, textMetadata.getEntityType());
        Assert.equals("5", textMetadata.get("text"));

        MetadataEntity actionMetadata = (MetadataEntity) processedMetadata.getTs("action");
        Assert.equals(MetadataEntityType.ACTION, actionMetadata.getEntityType());
        Assert.equals(LentaRecordType.PUBLIC_RESOURCE_OWNED.value(), actionMetadata.get("block-type"));
        Assert.equals(RESOURCE_OWNER_ID, actionMetadata.get("uid"));
        Assert.equals("GO_TO_LENTA", actionMetadata.get("action"));
        Assert.equals("456456456", actionMetadata.get("comment_id"));
        Assert.assertTrue(actionMetadata.fields.containsKeyTs("mtime"));

        MetadataEntity dateMetadata = (MetadataEntity)processedMetadata.getTs("date");
        Assert.equals(MetadataEntityType.DATE, dateMetadata.getEntityType());
        Assert.equals("1479142604421", dateMetadata.get("ms"));
        Assert.equals("14.11.2016", dateMetadata.get("text"));
    }

    private MetadataWrapper createMetaData(NotificationActor notificationActor) {
        MapF<String, MetadataEntity> metadata = Cf.hashMap();


        MapF<String, String> actorInfo = Cf.hashMap();
        actorInfo.put("uid", notificationActor.serialize());
        actorInfo.put("type", "user");
        metadata.put("actor", MetadataEntity.cons(actorInfo));

        MapF<String, String> commentInfo = Cf.hashMap();
        commentInfo.put("type", "comment");
        commentInfo.put("id", COMMENT_ID);
        commentInfo.put("text", "Вася дурак!");
        metadata.put("comment", MetadataEntity.cons(commentInfo));

        MapF<String, String> entityInfo = Cf.hashMap();
        entityInfo.put("type", "resource");
        entityInfo.put("id", RESOURCE_ID);
        entityInfo.put("name", "Some resource");
        entityInfo.put("resourcetype", "FILE");
        metadata.put("entity", MetadataEntity.cons(entityInfo));


        MapF<String, String> dateInfo = Cf.hashMap();
        dateInfo.put("type", "date");
        dateInfo.put("ms", "1479142604421");
        metadata.put("date", MetadataEntity.cons(dateInfo));


        MapF<String, String> actionInfo = Cf.hashMap();
        actionInfo.put("type", "action");
        actionInfo.put("block-type", LentaRecordType.PUBLIC_RESOURCE_OWNED.value());
        actionInfo.put("uid", RESOURCE_OWNER_ID);
        actionInfo.put("mtime", Long.toString(Instant.now().getMillis()));

        metadata.put("action", MetadataEntity.cons(actionInfo));

        return new MetadataWrapper(metadata);
    }

}
