package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public interface MpfsRelatedMetadataProcessor extends MetadataProcessor {
    MpfsClient getMpfsClient();

    int retryCount();

    Logger logger = LoggerFactory.getLogger(MpfsRelatedMetadataProcessor.class);

    /**
     * Obtains MpfsFileInfo for given ID, as cached in context.
     *
     * @param ownerAndResourceId resource identity as "owner-id:resource-id"
     * @param context            Cache to be used across different metadata processors
     * @return MpfsFileInfo object, or null if anything goes wrong.
     */
    default MpfsFileInfo getMpfsFileInfo(String ownerAndResourceId, MetadataProcessorContext context) {
        return context.getMpfsFileInfo(
                ownerAndResourceId,
                () -> {
                    MpfsResourceId resourceId = MpfsResourceId.parse(ownerAndResourceId);
                    return RetryUtils.retryOrThrow(logger, retryCount(),
                            () -> getMpfsFileInfoFromMpfs(resourceId.owner.getUid(), resourceId.fileId));
                });
    }

    // TODO: merge with upper method
    default MpfsFileInfo getMpfsFileInfo(Option<PassportUid> uid, String ownerAndResourceId, MetadataProcessorContext context) {
        String uidPrefix = uid.map(u -> u.toString() + "_").getOrElse("");
        return context.getMpfsFileInfo(
                uidPrefix + ownerAndResourceId,
                () -> {
                    MpfsResourceId resourceId = MpfsResourceId.parse(ownerAndResourceId);
                    return RetryUtils.retryOrThrow(logger, retryCount(),
                            () -> {
                                PassportUid ownerUid = resourceId.owner.getUid();
                                return getMpfsFileInfoFromMpfs(uid.getOrElse(ownerUid), ownerUid, resourceId.fileId);
                            });
                });
    }

    default MpfsFileInfo getMpfsFileInfo(Option<PassportUid> uid, String entityType, String entityId, MetadataProcessorContext context) {
        String uidPrefix = uid.map(u -> u.toString() + "_").getOrElse("");
        return context.getMpfsFileInfo(
                uidPrefix + entityId,
                () -> {
                    MpfsResourceId resourceId = MpfsResourceId.parse(entityId);
                    return RetryUtils.retryOrThrow(logger, retryCount(),
                            () -> {
                                PassportUid ownerUid = resourceId.owner.getUid();
                                return getMpfsFileInfoFromMpfs(uid.getOrElse(ownerUid), entityType, entityId);
                            });
                });
    }

    default MpfsFileInfo getPublicInfo(MpfsFileInfo mpfsFileInfo, MetadataProcessorContext context) {
        return context.getPublicInfo(mpfsFileInfo.getMeta().getFileId().get(),
                () -> {
                    String publicHash = mpfsFileInfo.getMeta().getPublicHash().get();
                    Option<MpfsFileInfo> fileInfo =
                            RetryUtils.retryOrThrow(logger, retryCount(), () -> getResourceInfoFromMpfs(publicHash));
                    return fileInfo.getOrNull();
                });
    }

    default Option<MpfsFileInfo> getResourceInfoFromMpfs(String publicHash) {
        return getMpfsClient().getFileInfoOByPrivateHash(publicHash);
    }

    default MpfsFileInfo getMpfsFileInfoFromMpfs(PassportUid owner, String resourceId) {
        return getMpfsClient().getFileInfoByFileId(MpfsUser.of(owner), resourceId);
    }

    default MpfsFileInfo getMpfsFileInfoFromMpfs(PassportUid uid, PassportUid owner, String fileId) {
        return getMpfsClient().getFileInfoByFileId(MpfsUser.of(uid), owner.toString(), fileId);
    }

    default MpfsFileInfo getMpfsFileInfoFromMpfs(PassportUid uid, String entityType, String entityId) {
        return getMpfsClient().getFileInfoByCommentId(MpfsUser.of(uid), entityType, entityId);
    }
}
