package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * @author buberman
 */
@RunWith(MockitoJUnitRunner.class)
public class MpfsRelatedMetadataProcessorTest {

    /**
     * Common run: call for MPFS value, returns a value
     */
    @Test
    public void testCommonRun() {
        MpfsClient mpfsClient = mock(MpfsClientImpl.class);

        when(mpfsClient.getFileInfoByFileId(any(), any()))
                .thenReturn(new MpfsFileInfo("name", "type", null));

        TestProcessor testProcessor = new TestProcessor(mpfsClient, 5);
        MetadataEntity field = new MetadataEntity();

        testProcessor.processField(field, null, new MetadataProcessorContext());
        assertEquals("name", field.get("value"));
    }

    @Test
    public void testRunWithRetries() {
        MpfsClient mpfsClient = mock(MpfsClientImpl.class);

        when(mpfsClient.getFileInfoByFileId(any(), any()))
                .thenThrow(NullPointerException.class)
                .thenThrow(NullPointerException.class)
                .thenReturn(new MpfsFileInfo("name", "type", null));

        TestProcessor testProcessor = new TestProcessor(mpfsClient, 5);
        MetadataEntity field = new MetadataEntity();

        testProcessor.processField(field, null, new MetadataProcessorContext());
        assertEquals("name", field.get("value"));
    }

    @Test(expected = NullPointerException.class)
    public void testInsufficientRetries() {
        MpfsClient mpfsClient = mock(MpfsClientImpl.class);

        when(mpfsClient.getFileInfoByFileId(any(), any()))
                .thenThrow(NullPointerException.class)
                .thenThrow(NullPointerException.class)
                .thenReturn(new MpfsFileInfo("name", "type", null));

        TestProcessor testProcessor = new TestProcessor(mpfsClient, 1);
        MetadataEntity field = new MetadataEntity();

        testProcessor.processField(field, null, new MetadataProcessorContext());
        assertEquals("name", field.get("value"));
    }


    private static class TestProcessor implements MpfsRelatedMetadataProcessor {
        private MpfsClient mpfsClient;
        private int retryCount;

        public TestProcessor(MpfsClient mpfsClient, int retryCount) {
            this.mpfsClient = mpfsClient;
            this.retryCount = retryCount;
        }

        @Override
        public ListF<MetadataEntityType> getProcessingTypes() {
            return null;
        }

        @Override
        public MpfsClient getMpfsClient() {
            return mpfsClient;
        }

        @Override
        public int retryCount() {
            return retryCount;
        }

        @Override
        public void processField(MetadataEntity field, NotificationRecord record, MetadataProcessorContext context) {
            MpfsFileInfo mpfsFileInfo = getMpfsFileInfo("12:qwqeryt", context);
            field.put("value", mpfsFileInfo.name.get());
        }
    }
}
