package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.chemodan.app.notifier.notification.disk.DiskGroups;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public class ResourceMetadataProcessor implements MpfsRelatedMetadataProcessor {
    private static final Logger logger = LoggerFactory.getLogger(ResourceMetadataProcessor.class);

    public static final int INF_EXPIRE_SECONDS = 2000000000;

    private final MpfsClient mpfsClient;
    private final int retryCount;

    public ResourceMetadataProcessor(MpfsClient mpfsClient, int retryCount) {
        this.mpfsClient = mpfsClient;
        this.retryCount = retryCount;
    }

    @Override
    public MpfsClient getMpfsClient() {
        return mpfsClient;
    }

    @Override
    public int retryCount() {
        return retryCount;
    }

    @Override
    public ListF<MetadataEntityType> getProcessingTypes() {
        return Cf.list(MetadataEntityType.RESOURCE);
    }

    @Override
    public void processField(MetadataEntity field, NotificationRecord record, MetadataProcessorContext context) {
        String ownerAndResourceId = field.get("id");
        Option<PassportUid> viewerUid = field.getO("uid").map(uid -> PassportUid.cons(Long.parseLong(uid)));
        logger.debug("Getting resource info for resource " + ownerAndResourceId);

        try {
            MpfsFileInfo fileInfo;
            if (isProperCommentsEntity(field, record)) {
                fileInfo = getMpfsFileInfo(viewerUid, field.get("entity_type"), ownerAndResourceId, context);
            } else {
                fileInfo = getMpfsFileInfo(viewerUid, ownerAndResourceId, context);
            }

            if (fileInfo != null) {
                field.put("text", fileInfo.name.get());
                field.put("resource_type", fileInfo.type.get());
                fileInfo.getMeta().getShortUrl().ifPresent(url -> field.put("short_url", url));

                if (fileInfo.getMeta().getPublicHash().isPresent()) {
                    try {
                        fileInfo.getMeta().getPmid().ifPresent(pmid -> {
                            String previewUrl = mpfsClient.generateZaberunUrl(
                                    pmid,
                                    fileInfo.name.get(),
                                    "preview",
                                    context.getUserId().map(DataApiUserId::serialize),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.of("S"),
                                    Option.empty(),
                                    Option.empty(),
                                    Option.of(INF_EXPIRE_SECONDS),
                                    Option.of(0),
                                    Option.of(true),
                                    Option.empty()
                            );
                            field.put(MetadataEntityNames.PREVIEW_URL, previewUrl + "&preview_type=jpg");
                        });
                    } catch (RuntimeException e) {
                        logger.error("Failed to add preview: {}", e);
                    }
                } else {
                    field.put(MetadataEntityNames.PREVIEW_URL,
                            fileInfo.getMeta().getPreview().map(p -> p + "&preview_type=jpg").getOrElse(""));
                }

                logger.debug("Resource info obtained successfully.");
            } else {
                logger.warn("Failed to obtain file info.");
            }

        } catch (RuntimeException e) {
            logger.error("Error while parsing resource data", e);
        }
    }

    private boolean isProperCommentsEntity(MetadataEntity field, NotificationRecord record) {
        return record != null
                && (record.type.getGroupName().equals(DiskGroups.COMMENTS)
                || record.type.getGroupName().equals(DiskGroups.LIKES))
                && StringUtils.isNotBlank(field.get("entity_type"));
    }
}
