package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
public class ResourceMetadataProcessorTest {

    @Mock
    private MpfsClient mpfsClient;

    private ResourceMetadataProcessor resourceProcessor;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);

        resourceProcessor = new ResourceMetadataProcessor(mpfsClient, 1);
    }

    @Test
    public void processPublicResource() {
        MpfsResourceId id = MpfsResourceId.parse("123:public-resource-id-123");
        MetadataEntity entityMetadata = createEntityMetadata(id.serialize());

        MpfsFileInfo fileInfo = createPublicFileMpfsFileInfo(id);
        MpfsFileInfo fileInfo2 = createPublicFileMpfsFileInfo2(id);

        when(mpfsClient.getFileInfoByFileId(MpfsUser.of(id.owner), id.owner.getRawValue(), id.fileId)).thenReturn(fileInfo);
        when(mpfsClient.getFileInfoOByPrivateHash(fileInfo.getMeta().getPublicHash().get()))
                .thenReturn(Option.of(fileInfo2));

        when(mpfsClient.generateZaberunUrl(any(), any(), any(), any(), any(), any(), any(), any(), any(),
                any(), any(), any(), any(), any(), any(), any(), any())).thenReturn("https://preview_zaberun_url.ru");

        resourceProcessor.processField(entityMetadata, null, new MetadataProcessorContext());


        Assert.equals(MetadataEntityType.RESOURCE, entityMetadata.getEntityType());
        Assert.equals(id.serialize(), entityMetadata.get("id"));
        Assert.equals("https://yadi.sk/shorturl", entityMetadata.get("short_url"));
        Assert.equals("Filename", entityMetadata.get("text"));
        Assert.equals("file", entityMetadata.get("resource_type"));
        Assert.equals("https://preview_zaberun_url.ru&preview_type=jpg", entityMetadata.get(MetadataEntityNames.PREVIEW_URL));
    }

    @Test
    public void processPrivateResource() {
        MpfsResourceId id = MpfsResourceId.parse("321:private-resource-id-123");
        MetadataEntity entityMetadata = createEntityMetadata(id.serialize());

        MpfsFileInfo fileInfo = createPrivateFileMpfsFileInfo(id);

        when(mpfsClient.getFileInfoByFileId(MpfsUser.of(id.owner), id.owner.getRawValue(), id.fileId)).thenReturn(fileInfo);

        resourceProcessor.processField(entityMetadata, null, new MetadataProcessorContext());

        Assert.equals(MetadataEntityType.RESOURCE, entityMetadata.getEntityType());
        Assert.equals(id.serialize(), entityMetadata.get("id"));
        Assert.equals("private_file.txt", entityMetadata.get("text"));
        Assert.equals("file", entityMetadata.get("resource_type"));
        Assert.equals("https://private-preview.ru&preview_type=jpg", entityMetadata.get(MetadataEntityNames.PREVIEW_URL));
    }

    private MpfsFileInfo createPrivateFileMpfsFileInfo(MpfsResourceId id) {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder()
                .resourceId(id)
                .source("disk")
                .pmid("fake_pmid")
                .fileId(id.fileId)
                .preview("https://private-preview.ru")
                .build();

        return new MpfsFileInfo("private_file.txt", "file", meta);
    }

    private MpfsFileInfo createPublicFileMpfsFileInfo(MpfsResourceId id) {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder()
                .resourceId(id)
                .source("disk")
                .fileId(id.fileId)
                .publicHash("publicHash1234567")
                .pmid("fake_pmid")
                .shortUrl("https://yadi.sk/shorturl")
                .shortNamed("https://yadi.sk/shortnamed")
                .build();

        return new MpfsFileInfo("Filename", "file", meta);
    }

    private MpfsFileInfo createPublicFileMpfsFileInfo2(MpfsResourceId id) {
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder()
                .resourceId(id)
                .source("disk")
                .pmid("fake_pmid")
                .fileId(id.fileId)
                .preview("https://preview.ru")
                .build();

        return new MpfsFileInfo("Filename", "file", meta);
    }


    private MetadataEntity createEntityMetadata(String id) {
        return new MetadataEntity(MetadataEntityType.RESOURCE, Cf.map("id", id));
    }

}
