package ru.yandex.chemodan.app.notifier.worker.metadataprocessor;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.notification.NotificationRecord;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author buberman
 */
public class TextMetadataProcessor implements MetadataProcessor {
    private static final Logger logger = LoggerFactory.getLogger(ActionMetadataProcessor.class);

    private final int maxLength;
    private final int maxLines;

    public TextMetadataProcessor(int maxLength, int maxLines) {
        this.maxLength = maxLength;
        this.maxLines = maxLines;
    }

    @Override
    public ListF<MetadataEntityType> getProcessingTypes() {
        return Cf.list(MetadataEntityType.TEXT);
    }

    @Override
    public void processField(MetadataEntity field, NotificationRecord record, MetadataProcessorContext context) {
        Option<String> text = field.getO("text");

        if (text.isPresent()) {
            String newText = abbreviateWithLines(text.get(), maxLines, maxLength);

            field.put("text", newText);
        }
    }

    /**
     * Abbreviates given string limiting it's length and line count.
     *
     * @param s      Text to abbreviate
     * @param lines  maximum number of lines, or -1 to skip line limitations
     * @param length maximum resulting text length, in characters, including line breaks, or -1 to skip length
     *               limitations
     */
    protected static String abbreviateWithLines(String s, int lines, int length) {
        if (length >= 0 && length < 4) {
            logger.warn("Trying to abbreviate with less than 4 characters length.");
            return s;
        }

        if (lines == 0) {
            logger.warn("Trying to abbreviate with less than 1 lines.");
            return s;
        }

        boolean abbreviated = false;

        String[] separateLines = s.split("\n");
        if (lines > 0 && separateLines.length > lines) {
            abbreviated = true;
            s = StringUtils.join(separateLines, "\n", 0, lines);
        }

        // CHEMODAN-48182
        // some symbols (i.e. emojis) are represented with more than one char so we'll look at codepoints during cut
        int codePointLength = s.codePointCount(0, s.length());

        abbreviated = abbreviated || (length > 0 && codePointLength > length);

        if (abbreviated) {
            if (length > 0 && codePointLength > length - 3) {
                s = s.substring(0, s.offsetByCodePoints(0, length - 3));
            }
            s += "...";
        }
        return s;
    }
}
