package ru.yandex.chemodan.app.notifier.worker.task;

import javax.annotation.Nullable;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.app.notifier.push.NotificationPushInfo;
import ru.yandex.chemodan.app.notifier.push.NotificationPushManager;
import ru.yandex.chemodan.app.notifier.settings.GlobalSubscriptionChannel;
import ru.yandex.chemodan.app.notifier.worker.metadata.MetadataEntityNames;
import ru.yandex.chemodan.app.notifier.worker.metadataprocessor.DiskMetadataProcessorManager;
import ru.yandex.chemodan.notifier.NotificationTaskQueueName;
import ru.yandex.chemodan.task.SendPushActiveUid;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author vpronto
 */
public class SendPushTask extends OnetimeTaskSupport<SendPushTask.Parameters> {

    private static final Logger logger = LoggerFactory.getLogger(SendPushTask.class);

    private NotificationPushManager notificationPushInfo;

    public SendPushTask(Parameters parameters) {
        super(parameters);
    }

    public SendPushTask(NotificationPushManager notificationPushInfo) {
        super(Parameters.class);
        this.notificationPushInfo = notificationPushInfo;
    }

    public SendPushTask(DataApiUserId uid, NotificationPushInfo info, MetadataWrapper metadata,
            Option<SetF<GlobalSubscriptionChannel>> channels, Instant deadline)
    {
        super(new Parameters(uid, info, metadata, channels, Option.of(deadline)));
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) {
        if (parameters.deadline.exists(deadline -> deadline.isBefore(Instant.now()))) {
            logger.info("Skip sending push {} for {} because deadline {} passed",
                    parameters.info, parameters.uid, parameters.deadline.get());
            return;
        }
        notificationPushInfo.consJob(parameters.uid, parameters.info, parameters.metadata)
                .push(parameters.channels);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return Parameters.Converter.class;
    }

    @Override
    public TaskQueueName queueName() {
        return NotificationTaskQueueName.NOTIFIER_REGULAR;
    }

    @BenderBindAllFields
    @Data
    public static class Parameters {
        public final DataApiUserId uid;
        public final NotificationPushInfo info;
        public final MetadataWrapper metadata;
        @BenderPart(wrapperName = "channels")
        public final Option<SetF<GlobalSubscriptionChannel>> channels;
        public final Option<Instant> deadline;

        public static class Converter implements ActiveUniqueIdentifierConverter<Parameters, SendPushActiveUid> {
            @Override
            public Class<SendPushActiveUid> getActiveUniqueIdentifierClass() {
                return SendPushActiveUid.class;
            }

            @Override
            public SendPushActiveUid convert(Parameters parameters) {
                Option<String> blockRecordIdO = parameters.getMetadata()
                        .getEntityField(MetadataEntityNames.ACTION, DiskMetadataProcessorManager.BLOCK_ID);
                Option<String> blockRecordTypeO = parameters.getMetadata()
                        .getEntityField(MetadataEntityNames.ACTION, DiskMetadataProcessorManager.BLOCK_TYPE);
                Option<ListF<String>> channelsO = parameters.channels
                        .map(channels -> channels.map(GlobalSubscriptionChannel::value).toList().sorted());
                Option<String> reqIdO = Option.empty();
                if (!blockRecordIdO.isPresent() || !blockRecordTypeO.isPresent()) {
                    String reqId = parameters.info.template.getTemplateSelectionReason().filterMap(r -> r.metadata.getO("reqid"))
                            .getOrElse(Random2.R.nextAlnum(10));
                    reqIdO = Option.of(reqId);
                    blockRecordIdO = Option.empty();
                    blockRecordTypeO = Option.empty();
                    channelsO = Option.empty();
                }
                return new SendPushActiveUid(parameters.uid.serialize(), reqIdO, blockRecordIdO, blockRecordTypeO, channelsO);
            }
        }
    }
}
