package ru.yandex.chemodan.app.orchestrator.admin;

import java.util.List;

import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerDbState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.manager.OrchestratorControl;
import ru.yandex.chemodan.app.orchestrator.manager.OrchestratorControlRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;

/**
 * @author friendlyevil
 */
@ActionContainer
public class LocationAdminPage {

    private final OrchestratorControl orchestratorControl;
    private final ContainersDao containersDao;
    private final OrchestratorControlRegistry orchestratorControlRegistry;

    public LocationAdminPage(OrchestratorControl orchestratorControl, ContainersDao containersDao,
                             OrchestratorControlRegistry orchestratorControlRegistry) {
        this.orchestratorControl = orchestratorControl;
        this.containersDao = containersDao;
        this.orchestratorControlRegistry = orchestratorControlRegistry;
    }


    @ZAction(file = "LocationAdminPage-index.xsl", defaultAction = true)
    @Path("/locations")
    public LocationPojo index() {
        return new LocationPojo(getStatisticByAllLocations(), orchestratorControlRegistry.get().getClosedDC());
    }

    @Path(value = "orchestrator/closeLocation", methods = HttpMethod.POST)
    public String closeLocation(@RequestParam("location") String location) {
        OrchestratorControlRegistry.ControlPojo controlPojo = orchestratorControlRegistry.get();
        if (controlPojo.getClosedDC().isPresent()) {
            return "ДЦ не будет закрыт: один из ДЦ уже закрыт (" + controlPojo.getClosedDC().get() + ")";
        }

        int minimumNumberOfContainersInLocation = orchestratorControl.getMinimumNumberOfContainersInLocation();
        if (getStatisticByAllLocations()
                .filterKeys(s -> !s.equals(location))
                .filterValues(s -> s.getAvailable() < minimumNumberOfContainersInLocation).isNotEmpty()) {
            return "ДЦ не будет закрыт: в некоторых ДЦ еще не поднялись контейнеры";
        }

        orchestratorControlRegistry.put(new OrchestratorControlRegistry.ControlPojo(Option.of(location)));
        return "ok";
    }

    @Path(value = "orchestrator/openLocation", methods = HttpMethod.POST)
    public boolean openLocation(@RequestParam("location") String location) {
        OrchestratorControlRegistry.ControlPojo controlPojo = orchestratorControlRegistry.get();
        if (controlPojo.getClosedDC().isPresent() && controlPojo.getClosedDC().get().equals(location)) {
            orchestratorControlRegistry.put(new OrchestratorControlRegistry.ControlPojo(Option.empty()));
            return true;
        }

        throw new IllegalArgumentException("DC " + location + " isn't closed");
    }


    private MapF<String, LocationStatistic> getStatisticByAllLocations() {
        MapF<String, ListF<Container>> containersByLocation = containersDao.getAll().groupBy(Container::getLocation);

        return orchestratorControl.getAllLocations().toMap(Function.identityF(),
                location -> calculateStatistic(containersByLocation.getOrDefault(location, Cf.list())));
    }

    private LocationStatistic calculateStatistic(ListF<Container> containers) {
        MapF<ContainerDbState, Integer> containersCountByStatus =
                containers.groupBy(Container::getState).mapValues(List::size);

        return new LocationStatistic(containers.size(),
                containersCountByStatus.getOrDefault(ContainerDbState.AVAILABLE, 0),
                containersCountByStatus.getOrDefault(ContainerDbState.DEACTIVATED, 0),
                containersCountByStatus.getOrDefault(ContainerDbState.LOST, 0));
    }

    @XmlRootElement(name = "content")
    @BenderBindAllFields
    @Value
    public static class LocationPojo {
        MapF<String, LocationStatistic> locations;
        Option<String> closedLocation;
    }

    @BenderBindAllFields
    @Value
    public static class LocationStatistic {
        int all;
        int available;
        int deactivated;
        int lost;
    }

}
