package ru.yandex.chemodan.app.orchestrator.cloud;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.UnknownHostException;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.manager.OrchestratorControl;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class ControlAgentClient {
    private static final Logger logger = LoggerFactory.getLogger(ControlAgentClient.class);

    private static final String HOST_FULL_STATE = "full";

    private final HttpClient httpClient;
    private final OrchestratorControl control;
    private final String nodeStateField;

    public ListF<String> listContainers(String host) {
        URI url = urlBuilder(host).appendPath("list").build();
        return ApacheHttpClientUtils.execute(new HttpGet(url), httpClient,
                new PodResponseHandler<>(ControlAgentClient::onListContainers));
    }

    public ListF<ContainerHostPortPojo> listEndpoints(String host) {
        URI url = urlBuilder(host).appendPath("list_endpoints").build();
        return ApacheHttpClientUtils.execute(new HttpGet(url), httpClient,
                new PodResponseHandler<>(ControlAgentClient::onListEndpoints));
    }

    public ContainerStateAndPortPojo getContainerState(String host, String containerId) {
        URI url = urlBuilder(host).appendPath("porto").appendPath(containerId).build();
        ContainerStateAndPortPojo state = wrapMissingHost(
                () -> ApacheHttpClientUtils.execute(new HttpGet(url), httpClient,
                        new PodResponseHandler<>(node -> ControlAgentClient.onContainerState(node, nodeStateField))),
                new ContainerStateAndPortPojo(ContainerNodeState.FAIL, ContainerPortoState.MISSING, 0));
        logger.info("Container {} state: {}", containerId, state);
        return state;
    }

    public boolean deleteContainer(String host, String containerId) {
        URI url = urlBuilder(host).appendPath("porto").appendPath(containerId).build();
        return wrapMissingHost(() -> ApacheHttpClientUtils.execute(
                new HttpDelete(url), httpClient, new PodResponseHandler<>(ControlAgentClient::onDeleteContainer)),
                true);
    }

    public Option<ContainerHostPortPojo> createContainer(String host) {
        URI url = urlBuilder(host).appendPath("new_container").build();
        return ApacheHttpClientUtils.execute(new HttpPost(url), httpClient,
                new PodResponseHandler<>(node -> ControlAgentClient.onCreateContainer(host, node)));
    }

    public boolean waitContainerAlive(Container container) {
        for (int intervalSecs: control.getContainerWaitIsAliveChecksRetryIntervals()) {
            if (isContainerAlive(container))
                return true;

            try {
                Thread.sleep(intervalSecs * 1000);
            } catch (InterruptedException e) {
                logger.error("waitContainerAlive sleep interrupted for container {}. Failure reason: {}",
                        container.getId(), e);
                throw ExceptionUtils.translate(e);
            }
        }

        return isContainerAlive(container);
    }

    public boolean isContainerAlive(Container container) {
        try {
            ContainerStateAndPortPojo containerState =
                    getContainerState(container.getPod().getHost().toString(), container.getId());
            return containerState.getPortoState() == ContainerPortoState.RUNNING && containerState.getNodeState() == ContainerNodeState.OK;
        } catch (RuntimeException e) {
            logger.error("Failed to resolve container {} state. Considered not alive. Failure reason: {}",
                    container.getId(), e);
            return false;
        }
    }

    private <T> T wrapMissingHost(Function0<T> handler, T resultOnMissing) {
        try {
            return handler.apply();
        } catch (RuntimeException e) {
            Throwable cause = e.getCause();
            if ((cause instanceof UnknownHostException) || (cause instanceof SocketTimeoutException)) {
                logger.warn("Host unreachable, container considered missing", cause);
                return resultOnMissing;
            } else {
                throw e;
            }
        }
    }

    @AllArgsConstructor
    private static class PodResponseHandler<T> extends Abstract200ExtendedResponseHandler<T> {
        private static final ObjectMapper mapper = new ObjectMapper();

        private final Function<JsonNode, T> nodeHandler;
        @Override
        protected T handle200Response(HttpResponse response) throws IOException {
            HttpEntity entity = response.getEntity();
            Check.notNull(entity);
            JsonNode jsonResponse = mapper.readValue(
                    InputStreamSourceUtils.wrap(entity.getContent()).readText("utf-8"),
                    JsonNode.class);
            return nodeHandler.apply(jsonResponse);
        }
    }

    static ListF<String> onListContainers(JsonNode node) {
        return Cf.x(node.elements()).map(JsonNode::textValue).toList();
    }

    static ListF<ContainerHostPortPojo> onListEndpoints(JsonNode node) {
        return Cf.x(node.fields()).toList().map(
                entry -> new ContainerHostPortPojo(HostPort.parse(entry.getKey()), entry.getValue().textValue()));
    }

    static ContainerStateAndPortPojo onContainerState(JsonNode node, String nodeStateField) {
        JsonNode nodeState = ObjectUtils.defaultIfNull(node.get(nodeStateField), node.get("office"));

        return new ContainerStateAndPortPojo(
                ContainerNodeState.valueOf(nodeState.textValue().toUpperCase()),
                ContainerPortoState.valueOf(node.get("state").textValue().toUpperCase()),
                node.get("port").intValue()
        );
    }

    static boolean onDeleteContainer(JsonNode node) {
        ContainerPortoState state = ContainerPortoState.valueOf(node.get("state").textValue().toUpperCase());
        return state == ContainerPortoState.DESTROYED;
    }

    static Option<ContainerHostPortPojo> onCreateContainer(String host, JsonNode node) {
        if (Option.ofNullable(node.get("state"))
                .map(JsonNode::textValue).map(String::toLowerCase).isSome(HOST_FULL_STATE))
        {
            return Option.empty();
        }

        String containerId = node.get("name").textValue();
        int port = node.get("port").intValue();
        return Option.of(new ContainerHostPortPojo(new HostPort(host, port), containerId));
    }

    private UriBuilder urlBuilder(String host) {
        return UriBuilder.cons("http://" + host);
    }

}
