package ru.yandex.chemodan.app.orchestrator.dao;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class ContainersDao {
    private static final RowMapper<Container> M = (rs, rowNum) -> new Container(
            rs.getString("id"),
            new HostPort(rs.getString("pod_host"), rs.getInt("service_port")),
            rs.getString("location"),
            Option.ofNullable(rs.getString("group_id")),
            ContainerDbState.R.valueOf(rs.getString("state")),
            new Instant(rs.getTimestamp("created_dt")),
            new Instant(rs.getTimestamp("modified_dt")),
            rs.getInt("sessions_count")
    );

    private final JdbcTemplate3 jdbcTemplate;

    public Option<Container> find(String containerId) {
        String query = "SELECT * FROM containers WHERE id = :id";
        MapF<String, Object> params = Cf.map("id", containerId);
        return jdbcTemplate.queryForOption(query, M, params);
    }

    public ListF<Container> find(SetF<String> containerIds) {
        if (containerIds.isEmpty()) {
            return Cf.list();
        }
        String query = "SELECT * FROM containers WHERE id IN (:ids)";
        MapF<String, Object> params = Cf.map("ids", containerIds);
        return jdbcTemplate.query(query, M, params);
    }

    public Option<Container> findAnyAvailableContainer(SetF<String> availableLocations, int maxSessions)
    {
        if (availableLocations.isEmpty()) {
            return Option.empty();
        }
        String query = "SELECT * FROM containers" +
                " WHERE state = 'available'" +
                " AND location IN (:available_locations)" +
                " AND sessions_count < :max_sessions_count" +
                " ORDER BY RANDOM() LIMIT 1";
        MapF<String, Object> params = Cf.map(
                "available_locations", availableLocations,
                "max_sessions_count", maxSessions
        );

        return jdbcTemplate.query(query, M, params).firstO();
    }

    public Option<Container> findAvailableContainer(Option<String> groupId,
                                                    SetF<String> availableLocations,
                                                    int maxSessions)
    {
        if (availableLocations.isEmpty()) {
            return Option.empty();
        }
        String query = "SELECT * FROM containers" +
                " WHERE state = 'available'" +
                " AND location IN (:available_locations)" +
                " AND group_id " + (groupId.isPresent() ? "= :group_id" : "IS NULL") +
                " AND sessions_count < :max_sessions_count" +
                " ORDER BY RANDOM() LIMIT 1";
        MapF<String, Object> params = Cf.map(
                "available_locations", availableLocations,
                "max_sessions_count", maxSessions
        ).plus(groupId.toMap(id -> Tuple2.tuple("group_id", id)));

        return jdbcTemplate.query(query, M, params).firstO();
    }

    public boolean setState(String containerId, ContainerDbState state) {
        String query = "UPDATE containers SET state = :state::container_state WHERE id = :id";
        MapF<String, Object> params = Cf.map("state", state.value(), "id", containerId);
        return jdbcTemplate.update(query, params) > 0;
    }

    public boolean setGroupIdForNewSession(String containerId, String groupId) {
        String query = "UPDATE containers SET group_id = :group_id, sessions_count = 1" +
                " WHERE id = :id AND group_id IS NULL";
        MapF<String, Object> params = Cf.map("group_id", groupId, "id", containerId);
        return jdbcTemplate.update(query, params) > 0;
    }

    public boolean incSessionsCount(String containerId, int maxSessions) {
        String query = "UPDATE containers SET sessions_count = sessions_count + 1" +
                " WHERE id = :id AND sessions_count < :max_sessions_count";
        MapF<String, Object> params = Cf.map("id", containerId, "max_sessions_count", maxSessions);
        return jdbcTemplate.update(query, params) > 0;
    }

    public boolean decSessionsCount(String containerId) {
        String query = "UPDATE containers SET sessions_count = sessions_count - 1" +
                " WHERE id = ? AND sessions_count > 0";
        return jdbcTemplate.update(query, containerId) > 0;
    }

    public boolean setSessionsCount(String containerId, int newValue, int oldValue) {
        String query = "UPDATE containers SET sessions_count = :new_value" +
                " WHERE id = :id AND sessions_count = :old_value";
        MapF<String, Object> params = Cf.map("id", containerId, "new_value", newValue, "old_value", oldValue);
        return jdbcTemplate.update(query, params) > 0;
    }

    public boolean create(Container container) {
        String q = "INSERT INTO containers " +
                "(id, pod_host, service_port, location, group_id, state, created_dt, modified_dt, sessions_count) " +
                "values (:id, :pod_host, :service_port, :location, :group_id, " +
                ":state::container_state, :created_dt, :modified_dt, :sessions_count)";
        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "id", container.getId(),
                "pod_host", container.getPod().getHost().toString(),
                "service_port", container.getPod().getPort(),
                "location", container.getLocation(),
                "group_id", container.getGroupId().getOrNull(),
                "state", container.getState().value(),
                "created_dt", container.getCreatedDt(),
                "modified_dt", container.getModifiedDt(),
                "sessions_count", container.getSessionsCount()
        ));
        return jdbcTemplate.update(q, params) > 0;
    }

    public ListF<Container> getAll() {
        return jdbcTemplate.query("SELECT * FROM containers", M);
    }

    public boolean delete(String containerId) {
        return jdbcTemplate.update("DELETE FROM containers WHERE id = ?", containerId) > 0;
    }
}
