package ru.yandex.chemodan.app.orchestrator.dao;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class SessionsDao {
    private static final RowMapper<Session> M = (rs, rowNum) -> new Session(
            rs.getString("id"),
            Option.ofNullable(rs.getString("container_id")),
            Option.ofNullable(rs.getString("finish_reason")).map(reason -> FinishReason.valueOf(reason.toUpperCase())),
            Option.ofNullable(rs.getTimestamp("finish_dt")).map(Instant::new),
            new Instant(rs.getTimestamp("created_dt")),
            new Instant(rs.getTimestamp("modified_dt")),
            new Instant(rs.getTimestamp("expiration_dt"))
    );

    private final JdbcTemplate3 jdbcTemplate;

    public Option<Session> find(String sessionId) {
        return jdbcTemplate.queryForOption("SELECT * FROM sessions WHERE id = ?", M, sessionId);
    }

    public int getCount() {
        return jdbcTemplate.queryForInt("SELECT COUNT (*) FROM sessions WHERE finish_dt IS NULL");
    }

    public ListF<Session> getAll(SqlLimits limits) {
        return jdbcTemplate.query("SELECT * FROM sessions WHERE finish_dt IS NULL" + limits.toMysqlLimits(), M);
    }

    public boolean create(Session session) {
        String q = "INSERT INTO sessions " +
                "(id, container_id, finish_reason, finish_dt, created_dt, modified_dt, expiration_dt) " +
                "values (:id, :container_id, :finish_reason, :finish_dt, :created_dt, :modified_dt, :expiration_dt)";
        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                "id", session.getId(),
                "container_id", session.getContainerId().getOrNull(),
                "finish_reason", session.getFinishReason().getOrNull(),
                "finish_dt", session.getFinishDt().getOrNull(),
                "created_dt", session.getCreatedDt(),
                "modified_dt", session.getModifiedDt(),
                "expiration_dt", session.getExpirationDt()
        ));
        return jdbcTemplate.update(q,params) > 0;
    }

    public boolean delete(String sessionId) {
        return jdbcTemplate.update("DELETE FROM sessions WHERE id = ?", sessionId) > 0;
    }

    public int deleteFinishedSessions(Instant expirationDt) {
        return jdbcTemplate.update("DELETE FROM sessions WHERE expiration_dt < ?", expirationDt);
    }

    public boolean touchSession(String sessionId, Instant expirationDt) {
        String query = "UPDATE sessions SET expiration_dt = :expiration_dt WHERE id = :id AND finish_dt IS NULL";
        MapF<String, Object> params = Cf.map("expiration_dt", expirationDt, "id", sessionId);
        return jdbcTemplate.update(query, params) > 0;
    }

    public boolean finishSession(String sessionId, SessionFinishReason finishReason) {
        String query =
                "UPDATE sessions SET finish_dt = :finish_dt, finish_reason = :finish_reason, container_id = NULL " +
                        "WHERE id = :id AND finish_dt IS NULL";
        MapF<String, Object> params = Cf.map(
                "finish_dt", Instant.now(),
                "finish_reason", finishReason.value(),
                "id", sessionId
        );
        return jdbcTemplate.update(query, params) > 0;
    }

    public ListF<String> findSessionsFromContainer(String containerId) {
        String query = "SELECT id FROM sessions WHERE container_id = ? AND finish_dt IS NULL";
        return jdbcTemplate.queryForList(query, String.class, containerId);
    }

    public int countSessionsFromContainer(String containerId) {
        String query = "SELECT COUNT (*) FROM sessions WHERE container_id = ? AND finish_dt IS NULL";
        return jdbcTemplate.queryForInt(query, containerId);
    }

    public ListF<String> findExpiredSessions(Instant expirationDt) {
        String query = "SELECT id FROM sessions WHERE expiration_dt < ? AND finish_dt IS NULL";
        return jdbcTemplate.queryForList(query, String.class, expirationDt);
    }
}
