package ru.yandex.chemodan.app.orchestrator.manager;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.env.EnvironmentType;

/**
 * @author yashunsky
 */
public class OrchestratorControl {
    private final String appKey = EnvironmentType.getActiveSecondary().map(env -> "-" + env).orElse("");

    private final DynamicProperty<ListF<String>> allLocations =
            new DynamicProperty<>("orchestrator-all-locations" + appKey, Cf.list("sas", "vla", "man"));

    private final DynamicProperty<Integer> minimumNumberOfContainersInLocation =
            new DynamicProperty<>("minimum-number-of-containers-in-location" + appKey, 300);

    private final DynamicProperty<Integer> sessionsPerContainer =
            new DynamicProperty<>("orchestrator-sessions-per-container" + appKey, 10);

    private final DynamicProperty<Integer> containersPerPod =
            new DynamicProperty<>("orchestrator-containers-per-pod" + appKey, 8);

    private final DynamicProperty<Integer> addContainerBatchSize =
            new DynamicProperty<>("orchestrator-add-container-batch-size" + appKey, 3);

    private final DynamicProperty<Integer> sessionsLimit =
            new DynamicProperty<>("orchestrator-sessions-limit" + appKey, 33000);

    private final DynamicProperty<Integer> sessionTtl =
            new DynamicProperty<>("orchestrator-session-ttl" + appKey + "[s]", 2400);

    private final DynamicProperty<Integer> sessionCleanPeriod =
            new DynamicProperty<>("orchestrator-clean-period" + appKey + "[s]", 1800);

    private final DynamicProperty<Integer> containerStatePollPause =
            new DynamicProperty<>("orchestrator-container-state-poll-pause" + appKey + "[s]", 1);

    private final DynamicProperty<Integer> containerActivationTtl =
            new DynamicProperty<>("orchestrator-container-activation-ttl" + appKey + "[s]", 240);

    private final DynamicProperty<ListF<Integer>> containerWaitIsAliveChecksRetryIntervals =
            new DynamicProperty<>("orchestrator-container_wait_is_alive_retry_intervals" + appKey + "[s]",
                    Cf.list(2, 4, 8, 16));

    private final DynamicProperty<Boolean> ignoreGroupId =
            new DynamicProperty<>("orchestrator-ignore-group-id" + appKey, false);

    private final OrchestratorControlRegistry orchestratorControlRegistry;

    public OrchestratorControl(OrchestratorControlRegistry orchestratorControlRegistry) {
        this.orchestratorControlRegistry = orchestratorControlRegistry;
    }

    public Instant getSessionExpirationDt() {
        return Instant.now().plus(Duration.standardSeconds(sessionTtl.get()));
    }

    public SetF<String> getEnabledLocations() {
        SetF<String> all = Cf.toHashSet(allLocations.get());
        orchestratorControlRegistry.get().getClosedDC().ifPresent(all::removeTs);
        return all;
    }

    public int getSessionsPerContainer() {
        return sessionsPerContainer.get();
    }

    public int getContainersPerPod() {
        return containersPerPod.get();
    }

    public int getSessionsLimit() {
        return sessionsLimit.get();
    }

    public int getCreateContainerBatchSize() {
        return addContainerBatchSize.get();
    }

    public Duration getContainerStatePollPause() {
        return Duration.standardSeconds(containerStatePollPause.get());
    }

    public Duration getSessionCleanPeriod() {
        return Duration.standardSeconds(sessionCleanPeriod.get());
    }

    public Duration getContainerActivationTtl() {
        return Duration.standardSeconds(containerActivationTtl.get());
    }

    public ListF<Integer> getContainerWaitIsAliveChecksRetryIntervals() {
        return containerWaitIsAliveChecksRetryIntervals.get();
    }

    public ListF<String> getAllLocations() {
        return allLocations.get();
    }

    public int getMinimumNumberOfContainersInLocation() {
        return minimumNumberOfContainersInLocation.get();
    }

    public boolean isIgnoreGroupId() {
        return ignoreGroupId.get();
    }
}
