package ru.yandex.chemodan.app.orchestrator.manager;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.orchestrator.cloud.ControlAgentClient;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerDbState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.dao.Session;
import ru.yandex.chemodan.app.orchestrator.dao.SessionFinishReason;
import ru.yandex.chemodan.app.orchestrator.dao.SessionsDao;
import ru.yandex.chemodan.app.orchestrator.exceptions.ContainerLostException;
import ru.yandex.chemodan.app.orchestrator.exceptions.NoAvailableContainersException;
import ru.yandex.chemodan.app.orchestrator.exceptions.SessionNotActiveException;
import ru.yandex.chemodan.app.orchestrator.exceptions.SessionsLimitException;
import ru.yandex.chemodan.app.orchestrator.pojo.SessionPojo;
import ru.yandex.chemodan.app.orchestrator.pojo.SessionsListPojo;
import ru.yandex.chemodan.app.orchestrator.unistat.EventsMetrics;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class SessionsManager {
    private static final Logger logger = LoggerFactory.getLogger(SessionsManager.class);

    private final OrchestratorControl control;
    private final ControlAgentClient controlAgentClient;

    private final SessionsDao sessionsDao;
    private final ContainersDao containersDao;

    public SessionPojo createSession(String sessionId, String groupId) {
        Option<Session> existingSessionO = sessionsDao.find(sessionId);

        // existing session

        if (existingSessionO.isPresent()) {
            Session existingSession = existingSessionO.get();
            if (!existingSession.isActive()) {
                EventsMetrics.notActiveSessions.inc();
                throw new SessionNotActiveException(sessionId);
            }

            String containerId = existingSession.getContainerId().get();

            Option<Container> containerO = containersDao.find(containerId);
            if (!containerO.isPresent()) {
                EventsMetrics.lostUsedContainers.inc();
                throw new ContainerLostException(containerId);
            }

            Container container = containerO.get();

            if (!controlAgentClient.isContainerAlive(container)) {
                EventsMetrics.lostUsedContainers.inc();
                throw new ContainerLostException(containerId);
            }

            Instant expirationDt = control.getSessionExpirationDt();
            if (sessionsDao.touchSession(sessionId, expirationDt)) {
                existingSession = existingSession.withExpirationDt(expirationDt);
            }
            EventsMetrics.touchedSessions.inc();
            return new SessionPojo(existingSession, container);
        }

        if (sessionsDao.getCount() >= control.getSessionsLimit()) {
            EventsMetrics.sessionsLimitAchieved.inc();
            throw new SessionsLimitException();
        }

        // new session
        return createSessionInAvailableGroupContainer(sessionId, groupId).orElseGet(
                () -> createSessionInAvailableFreeContainer(sessionId, groupId)
                        .getOrThrow(() -> {
                            EventsMetrics.noAvailableContainers.inc();
                            return new NoAvailableContainersException();
                        })
        );
    }

    private Option<SessionPojo> createSessionInAvailableGroupContainer(String sessionId, String groupId) {
        int maxSessionsPerContainer = control.getSessionsPerContainer();

        while (true) {
            Option<Container> containerO = findContainer(Option.of(groupId));
            if (!containerO.isPresent()) {
                return Option.empty();
            }

            Container container = containerO.get();

            if (!controlAgentClient.waitContainerAlive(container)) {
                EventsMetrics.lostUsedContainers.inc();
                logger.error("Used container {} suddenly found lost", container.getId());
                containersDao.setState(container.getId(), ContainerDbState.LOST);
            } else {
                boolean enoughCapacity = containersDao.incSessionsCount(container.getId(), maxSessionsPerContainer);
                if (enoughCapacity) {
                    EventsMetrics.containersReused.inc();
                    return Option.of(createSession(sessionId, container.getId(), groupId, container.getPod()));
                }
            }
        }
    }

    private Option<SessionPojo> createSessionInAvailableFreeContainer(String sessionId, String groupIdToSet) {
        while (true) {
            Option<Container> containerO = findContainer(Option.empty());
            if (!containerO.isPresent()) {
                return Option.empty();
            }

            Container container = containerO.get();

            if (!controlAgentClient.isContainerAlive(container)) {
                EventsMetrics.lostUnusedContainers.inc();
                logger.error("Free container {} suddenly found lost", container.getId());
                containersDao.setState(container.getId(), ContainerDbState.LOST);
            } else {
                boolean enoughCapacity = containersDao.setGroupIdForNewSession(container.getId(), groupIdToSet);
                if (enoughCapacity) {
                    EventsMetrics.containersLinkedToGroup.inc();
                    return Option.of(createSession(sessionId, container.getId(), groupIdToSet, container.getPod()));
                }
            }
        }
    }

    private Option<Container> findContainer(Option<String> groupId) {
        SetF<String> availableLocations = control.getEnabledLocations();
        int maxSessionsPerContainer = control.getSessionsPerContainer();

        Option<Container> container = containersDao.findAvailableContainer(groupId, availableLocations, maxSessionsPerContainer);
        if (container.isEmpty() && control.isIgnoreGroupId()) {
            container = containersDao.findAnyAvailableContainer(availableLocations, maxSessionsPerContainer);
        }
        return container;
    }

    private SessionPojo createSession(String sessionId, String containerId, String groupId, HostPort pod) {
        Instant now = Instant.now();
        Session session = new Session(
                sessionId, Option.of(containerId), Option.empty(), Option.empty(),
                now, now, control.getSessionExpirationDt());
        sessionsDao.create(session);
        EventsMetrics.createdSessions.inc();
        return new SessionPojo(session.getId(), pod, groupId, session.getExpirationDt());
    }

    public Option<SessionPojo> getSession(String sessionId) {
        return sessionsDao.find(sessionId).filterMap(session ->
                session.getContainerId().map(containerId -> {
                    Container container = containersDao.find(containerId).get();
                    return new SessionPojo(session, container);
                })
        );
    }

    public boolean touchSession(String sessionId) {
        EventsMetrics.touchedSessions.inc();
        return sessionsDao.touchSession(sessionId, control.getSessionExpirationDt());
    }

    public void finishSession(String sessionId, SessionFinishReason reason) {
        EventsMetrics.finishedSessions.inc(reason.value());
        if (reason != SessionFinishReason.FINISH_SIGNAL) {
            logger.error("Session {} finished with {} reason", sessionId, reason);
        }
        sessionsDao.find(sessionId).ifPresent(session -> {
            if (sessionsDao.finishSession(sessionId, reason)) {
                session.getContainerId().ifPresent(containersDao::decSessionsCount);
            }
        });
    }

    public SessionsListPojo getAllSessions(int offset, int limit) {
        int totalCount = sessionsDao.getCount();
        ListF<Session> sessions = sessionsDao.getAll(SqlLimits.range(offset, limit));

        MapF<String, Container> containers = containersDao
                .find(sessions.filterMap(Session::getContainerId).unique()).toMap(Container::getId, c -> c);

        ListF<SessionPojo> sessionPojos = sessions
                .zipWith(session -> session.getContainerId().filterMap(containers::getO))
                .filterBy2(Option::isPresent).map2(Option::get).map(SessionPojo::new);
        return new SessionsListPojo(totalCount, sessionPojos);
    }

    public void finishExpiredSessions() {
        sessionsDao.findExpiredSessions(Instant.now())
                .forEach(sessionId -> finishSession(sessionId, SessionFinishReason.EXPIRED));
    }

    public void finishSessionsFromLostContainer(String containerId) {
        sessionsDao.findSessionsFromContainer(containerId)
                .forEach(sessionId -> finishSession(sessionId, SessionFinishReason.CONTAINER_LOST));
    }

    public void deleteFinishedSessions() {
        EventsMetrics.deletedSessions.inc();
        sessionsDao.deleteFinishedSessions(Instant.now().minus(control.getSessionCleanPeriod()));
    }

    public int countSessionsFromContainer(String containerId) {
        return sessionsDao.countSessionsFromContainer(containerId);
    }
}
