package ru.yandex.chemodan.app.orchestrator.cloud;

import java.util.concurrent.TimeUnit;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.orchestrator.manager.OrchestratorControl;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.yp.discovery.YpDiscoveryClient;
import ru.yandex.yp.discovery.YpDiscoveryClientBuilder;
import ru.yandex.yp.discovery.YpDiscoveryInstance;
import ru.yandex.yp.discovery.model.EndpointsResolveRequest;
import ru.yandex.yp.discovery.model.EndpointsResolveResponse;
import ru.yandex.yp.discovery.model.YpEndpointResolveStatus;
import ru.yandex.yp.discovery.model.YpEndpointSet;

/**
 * @author yashunsky
 */
public class DiscoveryClient {
    private static final Logger logger = LoggerFactory.getLogger(DiscoveryClient.class);

    private final OrchestratorControl control;
    private final String serviceIdPrefix;
    private final Duration ypResponseTimeout;
    private final String clientName;

    public DiscoveryClient(OrchestratorControl control, String clientName, String serviceIdPrefix,
                           Duration ypResponseTimeout) {
        this.control = control;
        this.serviceIdPrefix = serviceIdPrefix;
        this.ypResponseTimeout = ypResponseTimeout;
        this.clientName = clientName;
    }

    public ListF<HostPort> resolveEndpoints() {
        return control.getEnabledLocations().flatMap(this::resolveClusterEndpoints);
    }

    public ListF<HostPort> resolveClusterEndpoints(String cluster) {
        EndpointsResolveRequest request = EndpointsResolveRequest.builder(cluster, getServiceId(cluster)).build();
        YpDiscoveryClient client = new YpDiscoveryClientBuilder(YpDiscoveryInstance.CROSS_DC, clientName).build();
        try {
            EndpointsResolveResponse response = client.serviceDiscoveryService().resolveEndpoints(request)
                    .get(ypResponseTimeout.getMillis(), TimeUnit.MILLISECONDS);

            YpEndpointResolveStatus status = response.getResolveStatus();
            if (status != YpEndpointResolveStatus.OK) {
                logger.warn("Endpoint problems in {}. Status: {}", cluster, status);
                return Cf.list();
            }

            return Cf.x(response.getEndpointSet()).flatMap(YpEndpointSet::getEndpoints)
                    .map(endpoint -> new HostPort(endpoint.getFqdn(), endpoint.getPort()));
        } catch (Exception e) {
            logger.error("Failed to resolve endpoints in " + cluster, e);
            throw ExceptionUtils.translate(e);
        } finally {
            if (!client.shutdown(1, TimeUnit.SECONDS)) {
                client.shutdownNow(100, TimeUnit.MILLISECONDS);
            }
        }
    }

    private String getServiceId(String cluster) {
        return serviceIdPrefix + "." + cluster;
    }
}
