package ru.yandex.chemodan.app.orchestrator.manager;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.orchestrator.cloud.ControlAgentClient;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.unistat.StateMetrics;
import ru.yandex.chemodan.concurrent.ExecutorUtils;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
abstract public class ActualizationManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(ActualizationManager.class);

    protected final ContainersDao containersDao;
    protected final ControlAgentClient controlAgentClient;
    protected final OrchestratorControl control;
    protected final BazingaTaskManager bazingaTaskManager;

    protected final ExecutorService actualizationService;

    public ActualizationManager(ContainersDao containersDao, ControlAgentClient controlAgentClient,
                                OrchestratorControl control, BazingaTaskManager bazingaTaskManager,
                                int actualizationThreads) {
        this.containersDao = containersDao;
        this.controlAgentClient = controlAgentClient;
        this.control = control;
        this.bazingaTaskManager = bazingaTaskManager;
        this.actualizationService = Executors.newFixedThreadPool(actualizationThreads);
    }

    public Option<Container> find(String containerId) {
        return containersDao.find(containerId);
    }

    abstract ListF<T> listItems();

    ListF<T> listSubItems(T item) {
        return Cf.list(item);
    }

    abstract void actualizeItem(T item);

    public ActualisationStatistic<T> actualize() {
        long start = System.nanoTime();
        ActualisationStatistic<T> statistic = actualizeItems(listItems());
        long end = System.nanoTime();
        String name = this.getClass().getSimpleName();
        int duration = (int) Duration.ofNanos(end).minusNanos(start).getSeconds();
        StateMetrics.actualizationDuration.set(duration, name + "_max");
        logger.info("{} actualization took {}s", name, duration);
        return statistic;
    }

    protected ActualisationStatistic<T> actualizeItems(ListF<T> items) {
        ActualisationStatistic<T> statistic;
        ListF<CompletableFuture<ActualisationStatistic<T>>> actualizations =
                items.map(item -> ExecutorUtils.submitWithYcridForwarding(
                () -> actualizeSubItemsAndGetFailures(item), actualizationService));

        try {
            ListF<ActualisationStatistic<T>> statistics = CompletableFutures.allOf(actualizations).get();
            statistic = ActualisationStatistic.of(statistics);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }

        return statistic;
    }

    protected ActualisationStatistic<T> actualizeSubItemsAndGetFailures(T item) {
        ActualisationStatistic<T> statistic = new ActualisationStatistic<>();
        ListF<T> subItems = Cf.list();
        try {
            subItems = listSubItems(item);
        } catch (Exception e) {
            logger.error("Error getting sub items for " + item, e);
            statistic.addGettingSublistFailed(new ActualisationFailure<>(item, e));
        }

        for (T subItem : subItems) {
            try {
                actualizeItem(subItem);
            } catch (Exception e) {
                logger.error("Error actualizing sub item " + subItem, e);
                statistic.addActualizeSubItemFailed(new ActualisationFailure<>(subItem, e));
            }
        }

        return statistic;
    }
}
