package ru.yandex.chemodan.app.orchestrator.manager;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.orchestrator.cloud.ContainerHostPortPojo;
import ru.yandex.chemodan.app.orchestrator.cloud.ContainerNodeState;
import ru.yandex.chemodan.app.orchestrator.cloud.ContainerPortoState;
import ru.yandex.chemodan.app.orchestrator.cloud.ContainerStateAndPortPojo;
import ru.yandex.chemodan.app.orchestrator.cloud.ControlAgentClient;
import ru.yandex.chemodan.app.orchestrator.cloud.DiscoveryClient;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerDbState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.tasks.onetime.ActivateContainerTask;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class FromCloudActualizationManager extends ActualizationManager<String> {
    private static final Logger logger = LoggerFactory.getLogger(FromCloudActualizationManager.class);
    private final DiscoveryClient discoveryClient;
    private final LocationResolver locationResolver;

    public FromCloudActualizationManager(ContainersDao containersDao, ControlAgentClient controlAgentClient,
                                         OrchestratorControl control, BazingaTaskManager bazingaTaskManager,
                                         DiscoveryClient discoveryClient, LocationResolver locationResolver,
                                         int actualizationThreads)
    {
        super(containersDao, controlAgentClient, control, bazingaTaskManager, actualizationThreads);
        this.discoveryClient = discoveryClient;
        this.locationResolver = locationResolver;
    }

    @Override
    ListF<String> listItems() {
        return control.getEnabledLocations().toList();
    }

    @Override
    ListF<String> listSubItems(String location) {
        return discoveryClient.resolveClusterEndpoints(location)
                .map(HostPort::getHost).map(Host::toString);
    }

    @Override
    void actualizeItem(String item) {
        addRunningContainersAtLocationIfMissing(item);
    }

    private void addRunningContainersAtLocationIfMissing(String podHost) {
        String location = locationResolver.resolveLocationFor(Option.of(podHost)).getDcName().get();
        logger.info("Actualizing pod {} in location {}", podHost, location);
        addContainersToPodIfNeeded(location, podHost);
    }

    private void addContainersToPodIfNeeded(String location, String podHost) {
        ListF<ContainerHostPortPojo> containers = controlAgentClient.listEndpoints(podHost);
        containers.forEach(container ->
                syncContainerStatusInDb(location, container.getContainerId(), container.getHostPort()));

        int containersCount = controlAgentClient.listEndpoints(podHost).length();
        int requiredContainers = control.getContainersPerPod() - containersCount;
        int batchSize = control.getCreateContainerBatchSize();
        int amount = Math.min(requiredContainers, batchSize);
        if (requiredContainers > 0) {
            logger.info("{} container required, adding {} containers (max batch size:{}) to {} @ {}",
                    requiredContainers, amount, batchSize, podHost, location);
        } else {
            logger.info("Found {} containers at {} @ {}, no more required", containersCount, podHost, location);
        }
        for (int i = 0; i < amount; i++) {
            Option<ContainerHostPortPojo> containerO = controlAgentClient.createContainer(podHost);
            if (!containerO.isPresent()) {
                logger.info("Could not create container @ {}", podHost);
            }
            containerO.ifPresent(container -> bazingaTaskManager.schedule(
                    new ActivateContainerTask(location, container.getContainerId(), container.getHostPort())));
        }
    }

    private boolean syncContainerStatusInDb(String location, String containerId, HostPort containerHostPort) {
        if (containersDao.find(containerId).isPresent()) {
            return true;
        }

        String host = containerHostPort.getHost().toString();

        ContainerStateAndPortPojo state = controlAgentClient.getContainerState(host, containerId);

        logger.info("Container {} state: {}", containerId, state);

        switch (state.getPortoState()) {
            case RUNNING:
                return state.getNodeState() == ContainerNodeState.OK
                        && createNewContainer(containerId, containerHostPort, location);
            case STARTING:
                return false;
            default:
                controlAgentClient.deleteContainer(host, containerId);
                return true;
        }
    }

    public boolean createNewContainer(String containerId, HostPort hostPort, String location) {
        Instant now = Instant.now();
        Container container = new Container(
                containerId, hostPort, location, Option.empty(), ContainerDbState.AVAILABLE, now, now, 0);
        return containersDao.create(container);
    }

    public boolean awaitStarted(String host, String containerId) {
        Instant start = Instant.now();
        while (true) {
            Duration activationTtl = control.getContainerActivationTtl();
            if (Instant.now().isAfter(start.plus(activationTtl))) {
                logger.error("Container {} could not start within ttl {}", containerId, activationTtl);
                return false;
            }

            ContainerStateAndPortPojo state = controlAgentClient.getContainerState(host, containerId);

            if (state.getPortoState() == ContainerPortoState.RUNNING) {
                if (state.getNodeState() == ContainerNodeState.OK) {
                    return true;
                }
            } else if (state.getPortoState() != ContainerPortoState.STARTING) {
                logger.error("Container {} is not starting properly. State: {}", containerId, state.getPortoState());
                return false;
            }

            ThreadUtils.sleep(control.getContainerStatePollPause());
        }
    }

    public void abortActivation(String host, String containerId) {
        controlAgentClient.deleteContainer(host, containerId);
    }
}
