package ru.yandex.chemodan.app.orchestrator.manager;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerLogicState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.dao.Session;
import ru.yandex.chemodan.app.orchestrator.dao.SessionsDao;
import ru.yandex.chemodan.app.orchestrator.unistat.StateMetrics;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */
public class StateManager extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(StateManager.class);

    private final String appKey = EnvironmentType.getActiveSecondary().map(env -> "-" + env).orElse("");

    private final DynamicProperty<Integer> maintenancePeriod =
            new DynamicProperty<>("orchestrator-state-maintenance-period" + appKey + "[s]", 60);

    private static final int TOP_N_GROUPS = 10;

    private static final ListF<String> ALL_LOCATIONS = Cf.list("sas", "iva", "myt", "vla", "man");

    private final SessionsDao sessionsDao;
    private final ContainersDao containersDao;
    private final OrchestratorControl control;

    public StateManager(SessionsDao sessionsDao, ContainersDao containersDao,
                        OrchestratorControl control)
    {
        this.sessionsDao = sessionsDao;
        this.containersDao = containersDao;
        this.control = control;

        setSleepBeforeFirstRun(false);
        setDelay(getMaintenancePeriod());
    }

    public void update() {
        logger.info("Updating state metrics");
        ListF<Container> containers = containersDao.getAll();

        MapF<Tuple2<String, ContainerLogicState>, Integer> containersBySateInDbPerDc =
                containers.countBy(c -> Tuple2.tuple(c.getLocation(), c.getLogicState()));

        ListF<ContainerLogicState> states = Cf.x(ContainerLogicState.values());

        MapF<Tuple2<String, ContainerLogicState>, Integer> containersBySatePerDc =
                ALL_LOCATIONS.flatMap(location -> states.map(state -> Tuple2.tuple(location, state)))
                        .zipWith(stateAndDc -> containersBySateInDbPerDc.getOrElse(stateAndDc, 0)).toMap();

        MapF<ContainerLogicState, Integer> containersBySateInDb = containers.countBy(Container::getLogicState);

        MapF<ContainerLogicState, Integer> containersBySate =
                Cf.x(ContainerLogicState.values()).zipWith(state -> containersBySateInDb.getOrElse(state, 0)).toMap();

        ListF<Session> sessions = sessionsDao.getAll(SqlLimits.all());

        int sessionsCount = sessions.size();
        MapF<String, Integer> sessionsByLocation = containers
                .groupBy(Container::getLocation).mapValues(cs -> cs.map(Container::getSessionsCount).sum(Cf.Integer));

        Tuple2List<String, Integer> topSessionsByGroup = containers
                .toTuple2List(Container::getGroupId, Container::getSessionsCount)
                .filterBy1(Option::isPresent).map1(Option::get)
                .groupBy1()
                .mapValues(sessionsCounts -> sessionsCounts.sum(Cf.Integer)).entries()
                .sortedBy2Desc().take(TOP_N_GROUPS);

        logger.info("State: containers by state: {}", containersBySate);
        logger.info("State: containers by state per DC: {}", containersBySatePerDc);
        logger.info("State: sessions count {}", sessionsCount);
        logger.info("State: sessions by dc {}", sessionsByLocation);
        logger.info("State: top groups {}", topSessionsByGroup);

        StateMetrics.sessionsCount.set(sessionsCount, "total_max");
        StateMetrics.sessionsCount.set(control.getSessionsLimit() - sessionsCount, "remaining_max");
        sessionsByLocation.forEach((location, count) -> StateMetrics.sessionsCount.set(count, location + "_max"));

        containersBySate.forEach((state, count) ->
                StateMetrics.containersByState.set(count, state.name().toLowerCase() + "_max"));
        containersBySatePerDc.forEach((locationAndState, count) ->
                StateMetrics.containersByState.set(
                        count, locationAndState._1, locationAndState._2.name().toLowerCase() + "_max"));

        topSessionsByGroup.forEach((groupId, count) -> StateMetrics.sessionsByGroup.set(count, groupId + "_max"));
        topSessionsByGroup.zipWithIndex().forEach((groupInfo, index) ->
                StateMetrics.sessionsByGroup.set(groupInfo._2, "top-" + (index + 1) + "_max"));
    }

    private Duration getMaintenancePeriod() {
        return Duration.standardSeconds(maintenancePeriod.get());
    }

    @Override
    protected void execute() throws Exception {
        if (maintenancePeriod.get() > 0) {
            MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_ASYNC_SS, this::update);
            setDelay(getMaintenancePeriod());
        }
    }
}
