package ru.yandex.chemodan.app.orchestrator.manager;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.orchestrator.cloud.ContainerPortoState;
import ru.yandex.chemodan.app.orchestrator.cloud.ControlAgentClient;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerDbState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerLogicState;
import ru.yandex.chemodan.app.orchestrator.dao.ContainersDao;
import ru.yandex.chemodan.app.orchestrator.tasks.onetime.DeactivateContainerTask;
import ru.yandex.chemodan.app.orchestrator.unistat.EventsMetrics;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public class ToCloudActualizationManager extends ActualizationManager<Container> {
    private static final Logger logger = LoggerFactory.getLogger(ToCloudActualizationManager.class);
    private final SessionsManager sessionsManager;

    public ToCloudActualizationManager(ContainersDao containersDao, ControlAgentClient controlAgentClient,
                                       OrchestratorControl control, BazingaTaskManager bazingaTaskManager,
                                       SessionsManager sessionsManager, int actualizationThreads)
    {
        super(containersDao, controlAgentClient, control, bazingaTaskManager, actualizationThreads);
        this.sessionsManager = sessionsManager;
    }

    @Override
    ListF<Container> listItems() {
        return containersDao.getAll();
    }

    @Override
    void actualizeItem(Container item) {
        actualizeContainer(item);
    }

    public boolean delete(String containerId) {
        return containersDao.delete(containerId);
    }

    private void actualizeContainer(Container container) {
        logger.info("Actualizing container {}", container.getId());
        boolean isAlive = controlAgentClient.isContainerAlive(container);

        int realSessionsCount = sessionsManager.countSessionsFromContainer(container.getId());
        if (realSessionsCount != container.getSessionsCount()) {
            container = container.withSessionsCount(realSessionsCount);
            if (containersDao.setSessionsCount(container.getId(), realSessionsCount, container.getSessionsCount())) {
                logger.info("Container {} session count set to {}", container.getId(), realSessionsCount);
            } else {
                logger.info("Failed to set container {} real session count. Will retry next time", container.getId());
            }
        }

        ContainerLogicState state = container.getLogicState();
        if (!control.getEnabledLocations().containsTs(container.getLocation())) {
            if (state == ContainerLogicState.READY) {
                logger.info("Container {} from disabled location {} is unused, shutting down",
                        container.getId(), container.getLocation());
                initiateContainerDeactivation(container, false);
                return;
            }
        }

        logger.info("Container {} logic state: {}, is alive: {}", container.getId(), state, isAlive);
        if (isAlive) {
            switch (state) {
                case READY:
                case ACTIVE:
                    break;
                case DIRTY:
                    initiateContainerDeactivation(container, false);
                    break;
                case DEACTIVATE:
                    scheduleContainerDeactivation(container);
                    break;
                case LOST:
                    reactivateContainer(container);
            }
        } else {
            switch (state) {
                case DEACTIVATE:
                    scheduleContainerDeactivation(container);
                    break;
                case READY:
                    EventsMetrics.lostUnusedContainers.inc();
                case DIRTY:
                    initiateContainerDeactivation(container, false);
                    break;
                case ACTIVE:
                    EventsMetrics.lostUsedContainers.inc();
                case LOST:
                    initiateContainerDeactivation(container, true);
                    break;
            }
        }
    }

    private void initiateContainerDeactivation(Container container, boolean finalizeSessions) {
        logger.info("Initiate container {} deactivation", container.getId());
        containersDao.setState(container.getId(), ContainerDbState.DEACTIVATED);

        if (finalizeSessions) {
            logger.error("Container {} was lost, finishing sessions", container.getId());
            sessionsManager.finishSessionsFromLostContainer(container.getId());
        }

        scheduleContainerDeactivation(container);
    }

    private void scheduleContainerDeactivation(Container container) {
        bazingaTaskManager.schedule(
                new DeactivateContainerTask(container.getPod().getHost().toString(), container.getId()));
    }

    private void reactivateContainer(Container container) {
        containersDao.setState(container.getId(), ContainerDbState.AVAILABLE);
    }

    public void initTerminationAndAwaitResult(String host, String containerId) {
        controlAgentClient.deleteContainer(host, containerId);
        while (true) {
            ContainerPortoState state = controlAgentClient.getContainerState(host, containerId).getPortoState();
            if (state == ContainerPortoState.STOPPED || state == ContainerPortoState.MISSING) {
                return;
            }
            ThreadUtils.sleep(control.getContainerStatePollPause());
        }
    }
}
