package ru.yandex.chemodan.app.orchestrator.tasks.onetime;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.orchestrator.dao.Container;
import ru.yandex.chemodan.app.orchestrator.dao.ContainerDbState;
import ru.yandex.chemodan.app.orchestrator.manager.ToCloudActualizationManager;
import ru.yandex.chemodan.app.orchestrator.tasks.OrchestratorQueueName;
import ru.yandex.chemodan.bazinga.TaskIdUtils;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
public class DeactivateContainerTask extends OnetimeTaskSupport<DeactivateContainerTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(DeactivateContainerTask.class);

    private final ToCloudActualizationManager actualizationManager;

    public DeactivateContainerTask(String host, String containerId) {
        super(new Parameters(host, containerId));
        this.actualizationManager = null;
    }

    public DeactivateContainerTask(ToCloudActualizationManager actualizationManager) {
        super(Parameters.class);
        this.actualizationManager = actualizationManager;
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        String containerId = parameters.containerId;
        Option<ContainerDbState> containerState = actualizationManager.find(containerId).map(Container::getState);
        if (!containerState.isPresent()) {
            logger.warn("No container {} found", containerId);
            return;
        }
        ContainerDbState state = containerState.get();
        if (state != ContainerDbState.DEACTIVATED) {
            logger.warn("No container {} status: {}. Deactivation aborted", containerId, state);
            return;
        }
        actualizationManager.initTerminationAndAwaitResult(parameters.host, containerId);
        actualizationManager.delete(containerId);
    }

    @Override
    public TaskId id() {
        return TaskIdUtils.withSecondaryEnv(super.id());
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(10);
    }

    @Override
    public TaskQueueName queueName() {
        return OrchestratorQueueName.DEACTIVATION;
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final String host;
        private final String containerId;
    }
}
