package ru.yandex.chemodan.balanceclient;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.balanceclient.model.method.BalanceClassMethodSpec;
import ru.yandex.chemodan.balanceclient.model.method.BalanceExtendedStatusMethodSpec;
import ru.yandex.chemodan.balanceclient.model.method.BalanceExtendedUntypedStatusMethodSpec;
import ru.yandex.chemodan.balanceclient.model.method.BalanceSimpleStatusMethodSpec;
import ru.yandex.chemodan.balanceclient.model.request.BalanceClientRequstFactory;
import ru.yandex.chemodan.balanceclient.model.request.CheckBindingRequest;
import ru.yandex.chemodan.balanceclient.model.request.CheckRequestPaymentRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateClientRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateInvoiceRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateOfferRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreatePersonRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateRequest2Item;
import ru.yandex.chemodan.balanceclient.model.request.CreateRequest2Request;
import ru.yandex.chemodan.balanceclient.model.request.CreateUserClientAssociationRequest;
import ru.yandex.chemodan.balanceclient.model.request.FindClientRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetBoundPaymentMethodsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetCardBindingURLRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetClientActsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetClientContractsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetClientPersonsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetOrdersInfoRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetPartnerBalanceRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetRequestChoicesRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetRequestPaymentMethodsRequest;
import ru.yandex.chemodan.balanceclient.model.request.ListClientPassportsRequest;
import ru.yandex.chemodan.balanceclient.model.request.ListPaymentMethodsSimpleRequest;
import ru.yandex.chemodan.balanceclient.model.request.PayRequestRequest;
import ru.yandex.chemodan.balanceclient.model.request.RemoveUserClientAssociationRequest;
import ru.yandex.chemodan.balanceclient.model.response.BalanceExtendedStatusResponse;
import ru.yandex.chemodan.balanceclient.model.response.CheckBindingResponse;
import ru.yandex.chemodan.balanceclient.model.response.CheckRequestPaymentResponse;
import ru.yandex.chemodan.balanceclient.model.response.ClientActInfo;
import ru.yandex.chemodan.balanceclient.model.response.ClientPassportInfo;
import ru.yandex.chemodan.balanceclient.model.response.CreateOfferResponse;
import ru.yandex.chemodan.balanceclient.model.response.CreateRequest2Response;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetBoundPaymentMethodsResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetCardBindingURLResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetClientPersonsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetOrdersInfoResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetPartnerBalanceContractResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetRequestChoicesResponse;
import ru.yandex.chemodan.balanceclient.model.response.ListPaymentMethodsSimpleResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.PayRequestResponse;
import ru.yandex.chemodan.balanceclient.model.response.RequestPaymentMethod;
import ru.yandex.inside.passport.PassportUid;

/**
 * Клиент для основного API баланса (описание методов: https://wiki.yandex-team.ru/balance/xmlrpc/)
 */

public class BalanceClient {
    /**
     * Мы используем Balance2.*, потому что это модно, современно и соответствует спеку (в отличие от Balance.*)
     */
    private static final BalanceExtendedStatusMethodSpec<Long> CREATE_CLIENT =
            new BalanceExtendedStatusMethodSpec<>("Balance2.CreateClient", Long.class);
    private static final BalanceSimpleStatusMethodSpec CREATE_USER_CLIENT_ASSOCIATION =
            new BalanceSimpleStatusMethodSpec("Balance2.CreateUserClientAssociation");
    private static final BalanceClassMethodSpec<CreateRequest2Response> CREATE_REQUEST_2 =
            new BalanceClassMethodSpec<>("Balance2.CreateRequest2", CreateRequest2Response.class);
    private static final BalanceClassMethodSpec<PayRequestResponse> PAY_REQUEST =
            new BalanceClassMethodSpec<>("Balance2.PayRequest", PayRequestResponse.class);
    private static final BalanceClassMethodSpec<CheckBindingResponse> CHECK_BINDING =
            new BalanceClassMethodSpec<>("Balance2.CheckBinding", CheckBindingResponse.class);
    private static final BalanceClassMethodSpec<GetClientPersonsResponseItem[]> GET_CLIENT_PERSONS =
            new BalanceClassMethodSpec<>("Balance2.GetClientPersons", GetClientPersonsResponseItem[].class);
    private static final BalanceClassMethodSpec<Long> CREATE_PERSON =
            new BalanceClassMethodSpec<>("Balance2.CreatePerson", Long.class);
    private static final BalanceClassMethodSpec<GetCardBindingURLResponse> GET_CARD_BINDING_URL =
            new BalanceClassMethodSpec<>("Balance2.GetCardBindingURL", GetCardBindingURLResponse.class);
    private static final BalanceSimpleStatusMethodSpec REMOVE_USER_CLIENT_ASSOCIATION =
            new BalanceSimpleStatusMethodSpec("Balance2.RemoveUserClientAssociation");
    private static final BalanceExtendedUntypedStatusMethodSpec<FindClientResponseItem[]> FIND_CLIENT =
            new BalanceExtendedUntypedStatusMethodSpec<>("Balance2.FindClient", FindClientResponseItem[].class);
    private static final BalanceClassMethodSpec<ListPaymentMethodsSimpleResponseItem> LIST_PAYMENT_METHODS_SIMPLE =
            new BalanceClassMethodSpec<>("Balance2.ListPaymentMethodsSimple",
                    ListPaymentMethodsSimpleResponseItem.class);
    private static final BalanceClassMethodSpec<CreateOfferResponse> CREATE_OFFER =
            new BalanceClassMethodSpec<>("Balance2.CreateOffer", CreateOfferResponse.class);
    private static final BalanceClassMethodSpec<GetClientContractsResponseItem[]> GET_CLIENT_CONTRACTS =
            new BalanceClassMethodSpec<>("Balance2.GetClientContracts", GetClientContractsResponseItem[].class);
    private static final BalanceClassMethodSpec<GetPartnerBalanceContractResponseItem[]> GET_PARTNER_BALANCE =
            new BalanceClassMethodSpec<>("Balance2.GetPartnerBalance", GetPartnerBalanceContractResponseItem[].class);
    private static final BalanceClassMethodSpec<GetOrdersInfoResponseItem[]> GET_ORDERS_INFO =
            new BalanceClassMethodSpec<>("Balance2.GetOrdersInfo", GetOrdersInfoResponseItem[].class);
    private static final BalanceClassMethodSpec<GetRequestChoicesResponse> GET_REQUEST_CHOICES =
            new BalanceClassMethodSpec<>("Balance2.GetRequestChoices", GetRequestChoicesResponse.class);
    private static final BalanceClassMethodSpec<Long> CREATE_INVOICE =
            new BalanceClassMethodSpec<>("Balance2.CreateInvoice", Long.class);
    private static final BalanceClassMethodSpec<RequestPaymentMethod[]> GET_REQUEST_PAYMENT_METHODS =
            new BalanceClassMethodSpec<>("Balance2.GetRequestPaymentMethods", RequestPaymentMethod[].class);
    private static final BalanceClassMethodSpec<CheckRequestPaymentResponse> CHECK_REQUEST_PAYMENT =
            new BalanceClassMethodSpec<>("Balance2.CheckRequestPayment", CheckRequestPaymentResponse.class);
    private static final BalanceClassMethodSpec<ClientPassportInfo[]> LIST_CLIENT_PASSPORTS =
            new BalanceClassMethodSpec<>("Balance2.ListClientPassports", ClientPassportInfo[].class);
    private static final BalanceClassMethodSpec<ClientActInfo[]> GET_CLIENT_ACTS =
            new BalanceClassMethodSpec<>("Balance2.GetClientActs", ClientActInfo[].class);
    private static final BalanceClassMethodSpec<GetBoundPaymentMethodsResponse[]> GetBoundPaymentMethods =
            new BalanceClassMethodSpec<>("Balance2.GetBoundPaymentMethods", GetBoundPaymentMethodsResponse[].class);


    private final BalanceXmlRpcClient balance;
    private final BalanceClientRequstFactory requestFactory;

    public BalanceClient(BalanceXmlRpcClient balance, BalanceClientRequstFactory requestFactory) {
        this.balance = balance;
        this.requestFactory = requestFactory;
    }

    public List<ClientActInfo> getClientActs(GetClientActsRequest request) {
        return Arrays.asList(balance.call(BalanceClient.GET_CLIENT_ACTS, request));
    }

    public GetBoundPaymentMethodsResponse[] getBoundPaymentMethods(GetBoundPaymentMethodsRequest request) {
        return balance.call(BalanceClient.GetBoundPaymentMethods, request);
    }

    /**
     * Создаёт клиента или обновляет существующего.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient
     *
     * @return идентификатор клиента (ClientId)
     */
    public long createClient(CreateClientRequest request) {
        Objects.requireNonNull(request.getOperatorUid(), "operatorUid");

        return balance.call(BalanceClient.CREATE_CLIENT, request).getData();
    }

    public GetOrdersInfoResponseItem[] getOrdersInfo(String contractId) {
        return balance.call(BalanceClient.GET_ORDERS_INFO, new GetOrdersInfoRequest(contractId));
    }

    /**
     * Создаёт клиента или обновляет существующего.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient
     *
     * @param operatorUid UID оператора
     * @return идентификатор клиента (ClientId)
     */
    public long createClient(long operatorUid, CreateClientRequest request) {
        request.setOperatorUid(operatorUid);

        return createClient(request);
    }

    public long createInvoice(CreateInvoiceRequest request) {
        return balance.call(CREATE_INVOICE, request);
    }

    public GetRequestChoicesResponse getPaymentMethods(Long operatorUid, long contractId, long requestId) {
        Objects.requireNonNull(operatorUid, "operatorUid");

        GetRequestChoicesRequest request = new GetRequestChoicesRequest()
                .withOperatorUid(operatorUid)
                .withContractId(contractId)
                .withRequestID(String.valueOf(requestId));
        return balance.call(GET_REQUEST_CHOICES, request);
    }

    public List<ClientPassportInfo> getClientRepresentativePassports(Long clientId, Option<Long> operatorUid) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");

        ListClientPassportsRequest request = new ListClientPassportsRequest()
                .withClientId(clientId);
        operatorUid.ifPresent(request::withOperatorUid);
        return Arrays.asList(balance.call(LIST_CLIENT_PASSPORTS, request));
    }

    /**
     * Создает заготовку счета
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createrequest2
     */
    public CreateRequest2Response createRequest(Long operatorUid, Long clientId, CreateRequest2Item createRequest2Item,
            boolean forceUnmoderated, Option<String> invoiceType)
    {
        return createRequest(operatorUid, clientId, createRequest2Item, forceUnmoderated, invoiceType,
                BalanceClient.CREATE_REQUEST_2);
    }

    public GetClientContractsResponseItem[] getClientContracts(GetClientContractsRequest request) {
        return balance.call(BalanceClient.GET_CLIENT_CONTRACTS, request);
    }

    public GetPartnerBalanceContractResponseItem[] getPartnerBalance(Integer serviceId, ListF<Long> contractIds) {
        return balance.call(GET_PARTNER_BALANCE, new GetPartnerBalanceRequest(serviceId, contractIds));
    }

    /**
     * Инициализация процесса оплаты реквеста
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.payrequest
     */
    public PayRequestResponse payRequest(PayRequestRequest request) {
        return balance.call(BalanceClient.PAY_REQUEST, request);
    }

    public CheckBindingResponse checkBinding(CheckBindingRequest request) {
        return balance.call(BalanceClient.CHECK_BINDING, request);
    }

    public List<GetClientPersonsResponseItem> getClientPersons(Long clientId) {
        GetClientPersonsRequest request = requestFactory.GetClientPersonsRequest().withClientId(clientId);
        return Arrays.asList(balance.call(BalanceClient.GET_CLIENT_PERSONS, request));
    }

    public Long createPerson(CreatePersonRequest request) {
        return balance.call(CREATE_PERSON, request);
    }

    public GetCardBindingURLResponse getCardBindingURL(GetCardBindingURLRequest request)
    {
        return balance.call(BalanceClient.GET_CARD_BINDING_URL, request);
    }

    /**
     * Создает представителя клиента
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createuserclientassociation
     *
     * @param operatorUid       UID оператора
     * @param clientId          Идентификатор клиента
     * @param representativeUid UID представителя
     */
    public void createUserClientAssociation(Long operatorUid, Long clientId, Long representativeUid) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(representativeUid, "representativeUid");

        CreateUserClientAssociationRequest request = new CreateUserClientAssociationRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withRepresentativeUid(representativeUid);
        balance.call(CREATE_USER_CLIENT_ASSOCIATION, request);
    }

    /**
     * Метод создаёт договор-оферту.
     * <p>
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.createoffer
     */

    public CreateOfferResponse createOffer(CreateOfferRequest request) {
        return balance.call(CREATE_OFFER, request);
    }

    /**
     * Удаляет представителя клиента
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.removeuserclientassociation
     *
     * @param operatorUid       UID оператора
     * @param clientId          Идентификатор клиента
     * @param representativeUid UID представителя
     */
    public void removeUserClientAssociation(Long operatorUid, Long clientId, Long representativeUid) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(representativeUid, "representativeUid");

        RemoveUserClientAssociationRequest request = new RemoveUserClientAssociationRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withRepresentativeUid(representativeUid);
        balance.call(REMOVE_USER_CLIENT_ASSOCIATION, request);
    }


    /**
     * Метод возвращает список клиентов по переданному запросу на поиск
     * <p>
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.findclient
     */
    public List<FindClientResponseItem> findClient(FindClientRequest request) {
        BalanceExtendedStatusResponse<FindClientResponseItem[]> response =
                balance.call(BalanceClient.FIND_CLIENT, request);
        return Arrays.asList(response.getData());
    }

    /**
     * Запрос методов платежей доступных пользователю:
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.listpaymentmethodssimple
     */
    public ListPaymentMethodsSimpleResponseItem listPaymentMethodsSimple(ListPaymentMethodsSimpleRequest request) {
        return balance.call(BalanceClient.LIST_PAYMENT_METHODS_SIMPLE, request);
    }

    public List<RequestPaymentMethod> getRequestPaymentMethodsForContract(PassportUid uid, String requestId,
            String contractId)
    {
        return Arrays
                .asList(balance.call(BalanceClient.GET_REQUEST_PAYMENT_METHODS, new GetRequestPaymentMethodsRequest()
                        .withOperatorUid(uid)
                        .withRequestId(requestId)
                        .withContractId(contractId)));
    }

    public CheckRequestPaymentResponse checkRequestPayment(
            Integer serviceId,
            String requestId,
            Option<String> transactionId,
            Long operatorUid
    ) {
        return balance.call(BalanceClient.CHECK_REQUEST_PAYMENT, new CheckRequestPaymentRequest()
                .withServiceId(serviceId)
                .withRequestId(requestId)
                .withTransactionIdO(transactionId)
                .withOperatorUid(operatorUid));
    }

    protected CreateRequest2Request createRequest2Request(Long operatorUid, Long clientId,
            CreateRequest2Item createRequest2Item, boolean forceUnmoderated, Option<String> invoiceType)
    {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(createRequest2Item, "createRequest2Item");

        CreateRequest2Request request = new CreateRequest2Request()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withItems(Collections.singletonList(createRequest2Item))
                .withForceUnmoderated(forceUnmoderated);
        invoiceType.ifPresent(request::withInvoiceType);
        return request;
    }

    private <T> T createRequest(Long operatorUid, Long clientId,
            CreateRequest2Item createRequest2Item, boolean forceUnmoderated, Option<String> invoiceType,
            BalanceClassMethodSpec<T> balanceClassMethodSpec)
    {
        return balance.call(balanceClassMethodSpec,
                createRequest2Request(operatorUid, clientId, createRequest2Item, forceUnmoderated, invoiceType));
    }
}
