package ru.yandex.chemodan.balanceclient;


import java.net.URL;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.logging.log4j.CloseableThreadContext;
import org.apache.ws.commons.util.NamespaceContextImpl;
import org.apache.xmlrpc.XmlRpcException;
import org.apache.xmlrpc.XmlRpcRequest;
import org.apache.xmlrpc.client.XmlRpcClient;
import org.apache.xmlrpc.client.XmlRpcClientConfigImpl;
import org.apache.xmlrpc.client.XmlRpcClientException;
import org.apache.xmlrpc.client.XmlRpcCommonsTransport;
import org.apache.xmlrpc.client.XmlRpcCommonsTransportFactory;
import org.apache.xmlrpc.client.XmlRpcTransport;
import org.apache.xmlrpc.common.TypeFactoryImpl;
import org.apache.xmlrpc.common.XmlRpcController;
import org.apache.xmlrpc.common.XmlRpcStreamConfig;
import org.apache.xmlrpc.common.XmlRpcWorkerFactory;
import org.apache.xmlrpc.parser.NullParser;
import org.apache.xmlrpc.parser.TypeParser;
import org.apache.xmlrpc.serializer.NullSerializer;
import org.apache.xmlrpc.serializer.TypeSerializer;
import org.joda.time.Duration;
import org.xml.sax.SAXException;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.balanceclient.exception.BalanceErrorCodeException;
import ru.yandex.chemodan.balanceclient.exception.BalanceException;
import ru.yandex.chemodan.balanceclient.model.BalanceErrorResponse;
import ru.yandex.chemodan.balanceclient.model.method.BaseBalanceMethodSpec;
import ru.yandex.chemodan.balanceclient.model.request.BalanceRpcRequestParam;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderXmlParser;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Stopwatch;

/**
 * Базовый клиент для XmlRpc API Баланса. Используется через хелперы в других классах пакета
 * <p>
 * ВАЖНО: XML-RPC API Баланса иногда нарушает спецификацию (http://xmlrpc.scripting.com/spec.html) а именно:
 * "The body of the response is a single XML structure, a <methodResponse>,
 * which can contain a single <params> which contains a single <param> which contains a single <value>."
 * <p>
 * Нарушение заключается в том, что возвращаемое значение (value) может быть не одно.
 * Текущая используемая реализация клиента из-за этого возвращает в качестве ответа только ПОСЛЕДНЕЕ <value>
 * в <param>, а все предыдущие отбрасывает.
 * <p>
 * Пока это никак не влияет на получаемые нами значения, так как отбрасываются только константные статусы,
 * а в случае возврата списка значений в Балансе используются массивы, но это нужно учитывать при разработке.
 * Сейчас основной маркер того, что возвращаемое значение может быть отброшено - слово "кортеж" в документации
 * к АПИ.
 * Например:
 * - в ручке Balance.CreateClient (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient) по документации
 * получаем кортеж из статуса, сообщения и ClientID. Значит ответ нужно обрабатывать так,
 * как будто мы получаем только ClientID.
 * - в ручке Balance.GetFirmCountryCurrency (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency)
 * по документации получаем список из статуса, сообщения и данных. Значит обрабатывать нужно все три элемента
 * списка
 * <p>
 * В сервисе Balance2 таких проблем пока замечено не было.
 * <p>
 * Тип {@link Long} не поддерживается на стороне баланса библиотекой. Поля с такой размерностью передаются строками.
 */

public class BalanceXmlRpcClient {
    private static final Logger DEFAULT_LOGGER = LoggerFactory.getLogger(BalanceXmlRpcClient.class);
    private static final BenderXmlParser<BalanceErrorResponse> ERROR_RESPONSE_XML_PARSER =
            Bender.xmlParser(BalanceErrorResponse.class);

    private XmlRpcClientConfigImpl clientConfig;
    private final XmlRpcClient client;
    private URL endpointUrl;
    private int retriesNum;
    private final Logger logger;
    private final BalanceXmlRpcClientConfig balanceClientConfig;

    public BalanceXmlRpcClient(BalanceXmlRpcClientConfig config) {
        balanceClientConfig = config;
        logger = StringUtils.isBlank(balanceClientConfig.getLoggerName()) ? DEFAULT_LOGGER :
                LoggerFactory.getLogger(balanceClientConfig.getLoggerName());

        client = new XmlRpcClient();
        client.setTypeFactory(new XmlRpcTypeNil(client));
        client.setConfig(clientConfig);
        updateClientConfig();

        client.setTransportFactory(new TVMXmlRpcCommonsTransportFactory(client, balanceClientConfig));

    }

    BalanceXmlRpcClient(BalanceXmlRpcClientConfig config, XmlRpcWorkerFactory workerFactory) {
        this(config);
        client.setWorkerFactory(workerFactory);
        retriesNum = config.getMaxRetries();
    }

    private void updateClientConfig() {
        clientConfig = getXmlRpcClientConfig(balanceClientConfig);
        client.setConfig(clientConfig);
        endpointUrl = clientConfig.getServerURL();
    }

    /**
     * Создать инстанс настроек XML-RPC клиента из настроек клиента Баланса
     */
    static XmlRpcClientConfigImpl getXmlRpcClientConfig(BalanceXmlRpcClientConfig config) {
        XmlRpcClientConfigImpl xmlRpcClientConfig = new XmlRpcClientConfigImpl();
        xmlRpcClientConfig.setServerURL(config.getServerUrl());
        xmlRpcClientConfig.setConnectionTimeout(Ints.saturatedCast(config.getConnectionTimeout().getMillis()));
        xmlRpcClientConfig.setReplyTimeout(Ints.saturatedCast(config.getRequestTimeout().getMillis()));
        if (config.getTimeZone() != null) {
            xmlRpcClientConfig.setTimeZone(config.getTimeZone());
        }
        return xmlRpcClientConfig;
    }

    /**
     * Выполнить обращение в XML-RPC Баланса и смаппить ответ в соответствующую методу структуру
     */
    public <R, T extends BalanceRpcRequestParam> R call(BaseBalanceMethodSpec<R> methodSpec, T requestParam) {
        updateClientConfig();
        Object response = retriedCall(methodSpec.getFullName(), requestParam.asArray(), requestParam.getTimeout(),
                requestParam.getMaxRetries());
        return parseResponse(response, methodSpec);
    }

    @VisibleForTesting
    <R> R parseResponse(Object response, BaseBalanceMethodSpec<R> methodSpec) {
        try {
            return methodSpec.convertResponse(response, clientConfig.getTimeZone());
        } catch (Exception e) {
            DEFAULT_LOGGER.error("Failed to convert balance call result", e);
            throw new BalanceException("Failed to convert balance call result", e);
        }
    }
    /**
     * Выполнить серию обращений к методу XML-RPC Баланса с заданным таимаутом на одно обращение и максимальным
     * количеством повторов. Вернуть результат первого успешного обращения.
     * <p>
     * Если таимаут или максимальное количество попыток не заданы, берутся значения по-умолчанию
     */
    Object retriedCall(String methodName, Object[] params, @Nullable Duration overrideTimeout,
                       @Nullable Integer overrideRetries) {
        int maxRetries = overrideRetries == null ? retriesNum : overrideRetries;
        for (int currentRetry = 1; ; currentRetry++) {
            boolean success = false;
            Object response = null;
            Stopwatch stopwatch = Stopwatch.createAndStart();
            try {
                response = rawCall(methodName, params, overrideTimeout);
                success = true;
                return response;
            } catch (Exception e) {
                if (currentRetry >= maxRetries) {
                    String msg = String.format("Method %s try #%s with params %s: failed at %s",
                            methodName, currentRetry, JsonUtils.toJson(params), endpointUrl
                    );
                    DEFAULT_LOGGER.error(msg, e);
                    throw buildException(methodName, params, e);
                }
            } finally {
                try (final CloseableThreadContext.Instance ignored = CloseableThreadContext.put("name", "balance")) {
                    String status = success ? "completed" : "failed";
                    String result = success ? JsonUtils.toJson(response) : "-";
                    logger.info("Method {} try #{} with params {}: {} at {}, with result {}; took {} ms",
                            methodName,
                            currentRetry, JsonUtils.toJson(params), status, endpointUrl,
                            result, stopwatch.millisDuration());
                }
            }
        }
    }

    /**
     * Выполнить непосредственное обращение к методу XML-RPC Баланса с заданным таимаутом и вернуть ответ
     * <p>
     * Если таимаут не задан, берется таимаут по-умолчанию
     */
    Object rawCall(String methodName, Object[] params, @Nullable Duration timeout) throws XmlRpcException {
        Object response;
        if (timeout == null || clientConfig.getReplyTimeout() == timeout.getMillis()) {
            response = client.execute(methodName, params);
        } else {
            XmlRpcClientConfigImpl newConfig = clientConfig.cloneMe();
            newConfig.setReplyTimeout(Ints.saturatedCast(timeout.getMillis()));
            response = client.execute(newConfig, methodName, params);
        }

        return response;
    }

    private static class TVMXmlRpcCommonsTransportFactory extends XmlRpcCommonsTransportFactory {
        private final BalanceXmlRpcClientConfig configuration;

        public TVMXmlRpcCommonsTransportFactory(XmlRpcClient pClient, BalanceXmlRpcClientConfig configuration) {
            super(pClient);
            this.configuration = configuration;
        }

        @Override
        public HttpClient getHttpClient() {
            HttpClient httpClient = new HttpClient();
            if (configuration.getProxy().isPresent()) {
                HostConfiguration hostConfiguration = new HostConfiguration();
//                hostConfiguration.setProxyHost(configuration.getProxy().get()); TODO for recording
                httpClient.setHostConfiguration(hostConfiguration);
            }
            return httpClient;
        }

        @Override
        public XmlRpcTransport getTransport() {
            return new TVMAwareXmlRpcTransport(this, configuration);
        }
    }

    private static class TVMAwareXmlRpcTransport extends XmlRpcCommonsTransport {
        private final Tvm2 tvm2;
        private final Function<String, Option<Integer>> dstClientIdsResolver;
        private String host;

        public TVMAwareXmlRpcTransport(XmlRpcCommonsTransportFactory pFactory,
                                       BalanceXmlRpcClientConfig configuration) {
            super(pFactory);
            this.tvm2 = configuration.getTvm2();
            this.dstClientIdsResolver = configuration.getDstClientIdsResolver();
            this.host = configuration.getServerUrl().getHost();
        }

        @Override
        protected void initHttpHeaders(XmlRpcRequest pRequest) throws XmlRpcClientException {
            super.initHttpHeaders(pRequest);

            Option<Integer> dstClientId = dstClientIdsResolver.apply(host);
            if (!dstClientId.isPresent()) {
                DEFAULT_LOGGER.warn("Unable to find client id by host {}", host);
            } else {
                tvm2.getServiceTicket(dstClientId.get())
                        .ifPresent(ticket -> setRequestHeader(TvmHeaders.SERVICE_TICKET, ticket));
            }
        }
    }

    /**
     * Workaround for
     * "Failed to parse server's response: Unknown type: nil"
     */
    private static class XmlRpcTypeNil extends TypeFactoryImpl {

        XmlRpcTypeNil(XmlRpcController pController) {
            super(pController);
        }

        public TypeParser getParser(XmlRpcStreamConfig pConfig, NamespaceContextImpl pContext, String pURI,
                                    String pLocalName) {
            if (NullSerializer.NIL_TAG.equals(pLocalName) || NullSerializer.EX_NIL_TAG.equals(pLocalName)) {
                return new NullParser();
            } else {
                return super.getParser(pConfig, pContext, pURI, pLocalName);
            }
        }

        public TypeSerializer getSerializer(XmlRpcStreamConfig pConfig, Object pObject) throws SAXException {
            if (pObject instanceof XmlRpcTypeNil) {
                return new NullSerializer();
            } else {
                return super.getSerializer(pConfig, pObject);
            }
        }
    }

    private BalanceException buildException(String methodName, Object[] params, Exception e) {
        String msg = String.format("Method %s with params %s: failed at %s, with exception '%s'",
                methodName, JsonUtils.toJson(params), endpointUrl, e.getMessage());
        Option<BalanceErrorResponse> balanceParsedError = tryParseError(e.getMessage());
        if (balanceParsedError.isPresent() && balanceParsedError.get().getCodes().isNotEmpty()) {
            return new BalanceErrorCodeException(balanceParsedError.get().getCodes(),
                    balanceParsedError.get().getMessage(), msg, e);
        } else {
            return new BalanceException(msg, e);
        }
    }

    private static Option<BalanceErrorResponse> tryParseError(String message) {
        try {
            BalanceErrorResponse errorResponse = ERROR_RESPONSE_XML_PARSER.parseXml(message);
            return Option.ofNullable(errorResponse);
        } catch (Exception e) {
            DEFAULT_LOGGER.warn("Can't parse error message as xml response", e);
            return Option.empty();
        }
    }
}
