package ru.yandex.chemodan.balanceclient;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonLocation;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.type.TypeFactory;

import ru.yandex.commune.json.jackson.JodaTimeModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;

public class JsonUtils {
    private static final String DESERIALIZATION_ERROR = "can not deserialize object from json";

    public static final String DATETIME_PATTERN = "yyyy-MM-dd'T'HH:mm:ss.SSSSSS";

    public static final ObjectMapper MAPPER = new ObjectMapper()
            .registerModule(new BoltsModule())
            .registerModule(new JodaTimeModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    private static final ObjectWriter DETERMINISTIC_WRITER = new ObjectMapper()
            .writer().with(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS);

    private JsonUtils() {
        // No instantiating, please
    }

    public static ObjectMapper getObjectMapper() {
        return MAPPER;
    }

    public static TypeFactory getTypeFactory() {
        return MAPPER.getTypeFactory();
    }

    /**
     * Десериализует объект из JSON
     *
     * @param json строка JSON
     * @param type тип объекта
     * @param <T>  тип объекта
     * @return десериализованный объект
     */
    public static <T> T fromJson(String json, Class<T> type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(byte[] json, Class<T> type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(byte[] json, JavaType valueType) {
        try {
            return MAPPER.readValue(json, valueType);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(InputStream inputStream, Class<T> type) {
        try {
            return MAPPER.readValue(inputStream, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(String json, JavaType type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(String json, TypeReference<T> typeReference) {
        try {
            return MAPPER.readValue(json, typeReference);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(JsonNode jsonNode, Class<T> type) {
        try {
            return MAPPER.treeToValue(jsonNode, type);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not parse json", e);
        }
    }

    public static JsonNode fromJson(String json) {
        try {
            return MAPPER.readTree(json);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not parse json", e);
        }
    }


    /**
     * Сериализует объект в JSON
     *
     * @param obj объект
     * @return строка JSON
     */
    public static String toJson(Object obj) {
        try {
            return MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    public static byte[] toJsonBytes(Object obj) {
        try {
            return MAPPER.writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    public static byte[] toJsonBytesForType(JavaType type, Object obj) {
        try {
            return MAPPER.writerFor(type).writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    public static String toJsonNullable(Object obj) {
        return obj != null ? toJson(obj) : null;
    }

    /**
     * Преобразует коллекцию в JSON массив. null остается null, пустая коллекция также преобразуется в null
     * Если пустая коллекция не дожна преобразовываться в null, то можно использовать {@link #toJsonNullable(Object)}
     *
     * @param collection колекция для конвертирования в json
     * @return json-строка или null
     */
    public static String toJsonCollectionEmptyToNull(@Nullable Collection<?> collection) {
        return collection == null || collection.isEmpty() ? null : toJson(collection);
    }

    /**
     * Преобразует объект в JSON строку в повторяемом виде.
     * Это не такой канонический вид, к которому мы привыкли в перле, т.к. ключи объекта выводятся в том порядке,
     * в котором они определены в классе.
     * Но хотя бы сортируются ключи хеш мапы, что делает результирующий JSON хотя бы предсказуемым.про
     */
    public static String toDeterministicJson(Object obj) {
        try {
            return DETERMINISTIC_WRITER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    /**
     * Извлекает координаты ошибки из JsonMappingException и преобразует в human-readable сообщение.
     * {@link Throwable#getMessage()} при этом игнорируется.
     */
    public static String extractJsonMappingErrorLocation(JsonMappingException ex) {
        List<JsonMappingException.Reference> referenceList = ex.getPath();
        StringJoiner joiner = new StringJoiner(" -> ", "[", "]");
        for (JsonMappingException.Reference ref : referenceList) {
            String fieldName = ref.getFieldName();
            if (fieldName != null) {
                joiner.add(fieldName);
            } else {
                final Object srcObject = ref.getFrom();
                if (srcObject instanceof List) {
                    joiner.add(String.format("array[%d]", ((List) srcObject).size()));
                } else {
                    if (srcObject instanceof Map) {
                        joiner.add("object");
                    }
                }
            }
        }
        JsonLocation location = ex.getLocation();
        return String.format("can't map json into internal data model; path: %s, line: %d, column: %d",
                joiner.toString(), location.getLineNr(), location.getColumnNr());
    }

    public static <T> T ifNotNull(JsonNode jsonNode, Function<JsonNode, T> converter) {
        if (jsonNode == null) {
            return null;
        }

        return jsonNode.isNull() ? null : converter.apply(jsonNode);
    }
}
