package ru.yandex.chemodan.balanceclient.model.request;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.MissingNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import lombok.Getter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.balanceclient.JsonUtils;

/**
 * Запрос для инициализации оплаты
 * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.payrequest
 */
@Getter
@JsonInclude(JsonInclude.Include.NON_NULL)
public class PayRequestRequest extends BalanceRpcRequestParam {
    @JsonIgnore
    private final String RECURRING_PAYMENT_METHOD = "recurring";

    @JsonIgnore
    private Long operatorUid;

    @JsonProperty("RequestID")
    private String requestId;

    @JsonProperty("PaymentMethodID")
    private String paymentMethodId;

    @JsonProperty("Currency")
    private String currency;

    @JsonProperty("PersonID")
    private String personId;

    @JsonProperty("RedirectURL")
    private String redirectUrl;

    @JsonProperty("ContractID")
    private String contractId;

    @JsonProperty("PaymentMode")
    private String paymentMode;

    @JsonProperty("NotificationURL")
    private String notificationUrl;

    @JsonProperty("Payload")
    private String payload;

    @JsonIgnore
    private MapF<String, String> payloadMap = Cf.hashMap();

    public PayRequestRequest withOperatorUid(Long operatorUid) {
        this.operatorUid = operatorUid;
        return this;
    }

    public PayRequestRequest withRequestId(String requestId) {
        this.requestId = requestId;
        return this;
    }

    public PayRequestRequest withPaymentMethodId(String paymentMethodId) {
        this.paymentMethodId = paymentMethodId;
        return this;
    }

    public PayRequestRequest withCurrency(String currency) {
        this.currency = currency;
        return this;
    }

    public PayRequestRequest withPersonId(Long personId) {
        this.personId = personId.toString();
        return this;
    }

    public PayRequestRequest withRedirectUrl(String redirectUrl) {
        this.redirectUrl = redirectUrl;
        return this;
    }

    public PayRequestRequest withContractId(String contractId) {
        this.contractId = contractId;
        return this;
    }

    public PayRequestRequest withNotificationUrl(String notificationUrl) {
        this.notificationUrl = notificationUrl;
        return this;
    }

    public PayRequestRequest withPayload(String payload) {
        this.payload = payload;
        return this;
    }

    public PayRequestRequest addToPayload(String name, Option<String> value) {
        value.ifPresent(v -> addToPayload(name, v));
        return this;
    }

    public PayRequestRequest addToPayload(String name, String value) {
        this.payloadMap.put(name, value);
        return this;
    }

    public PayRequestRequest asRecurrentPayment() {
        paymentMode = RECURRING_PAYMENT_METHOD;
        return this;
    }

    @Override
    public Object[] asArray() {
        if (payload != null) {
            JsonNode parent = JsonUtils.fromJson(payload);
            if (!(parent instanceof ObjectNode)) {
                throw new IllegalArgumentException("Payload must be json object");
            }

            for (String key : payloadMap.keys()) {
                JsonNode childNode = parent.findPath(key);
                if (childNode instanceof MissingNode) {
                    ((ObjectNode) parent).set(key, new TextNode(payloadMap.getOrThrow(key)));
                }
            }
            payload = parent.toString();
        } else if (payloadMap.isNotEmpty()) {
            payload = JsonUtils.toJson(payloadMap);
        }
        return new Object[]{operatorUid.toString(), objectToMap(this)};
    }

    public String getRequestId() {
        return requestId;
    }

    public String getPaymentMethodId() {
        return paymentMethodId;
    }
}


