package ru.yandex.chemodan.app.psbilling.core.admin;

import lombok.AllArgsConstructor;
import org.intellij.lang.annotations.Language;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.chemodan.balanceclient.model.response.GetBoundPaymentMethodsResponse;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.PaymentRequest;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@AllArgsConstructor
public class CardBinderScript {
    private static final Logger logger = LoggerFactory.getLogger(CardBinderScript.class);

    private final TrustClient trustClient;
    private final BalanceService balanceService;
    private final GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;
    private final CardDao cardDao;
    private final JdbcTemplate3 jdbcTemplate;

    public void bindCards(String trustUrl, int limit) {
        if (!trustUrl.isEmpty()) {
            trustClient.setUrl(trustUrl);
        }

        @Language("SQL") String sql = "with\n" +
                "     users_with_cards as (\n" +
                "         select distinct uid from cards where status = 'active'\n" +
                "     ),\n" +
                "     prepaid_uids as\n" +
                "         (\n" +
                "             select distinct (g.payment_info ->> 'uid') as uid\n" +
                "             from groups g\n" +
                "                      join group_services gs on g.id = gs.group_id\n" +
                "                      join group_products gp on gs.group_product_id = gp.id\n" +
                "             where gp.hidden = false\n" +
                "               and gp.payment_type = 'prepaid'::payment_type\n" +
                "         ),\n" +
                "     last_success_payments as (\n" +
                "         select distinct on (uid) uid, pr.request_id, pr.transaction_id\n" +
                "         from prepaid_uids pu\n" +
                "                  join group_trust_payment_requests pr on pr.operator_uid = pu.uid\n" +
                "         where pr.status = 'success'\n" +
                "         order by uid, pr.created_at desc, pr.request_id, pr.transaction_id\n" +
                "     )\n" +
                "select lp.*\n" +
                "from last_success_payments lp\n" +
                "left join users_with_cards uwc on uwc.uid = lp.uid\n" +
                "where uwc.uid is null\n" +
                "limit ?;";

        ListF<Tuple2<String, String>> lastSuccessPrepaidPayments =
                jdbcTemplate.query(sql, (rs, rowNum) ->
                                Tuple2.tuple(rs.getString("request_id"),
                                        rs.getString("transaction_id")),
                        limit);

        for (Tuple2<String, String> payment : lastSuccessPrepaidPayments) {
            try {
                logger.info("processing payment {}", payment);
                String requestId = payment._1;
                String transactionId = payment._2;

                Option<GroupTrustPaymentRequest> paymentRequestO =
                        groupTrustPaymentRequestDao.findByRequestId(requestId);
                if (!paymentRequestO.isPresent()) {
                    logger.error("The request {} is not found in DB", requestId);
                    continue;
                }
                GroupTrustPaymentRequest paymentRequest = paymentRequestO.get();
                if (paymentRequest.getStatus() != PaymentRequestStatus.SUCCESS) {
                    logger.error("not success status for {}", paymentRequest);
                    continue;
                }

                PassportUid uid = PassportUid.cons(Long.parseLong(paymentRequest.getOperatorUid()));

                PaymentResponse paymentResponse = trustClient.getPayment(PaymentRequest.builder()
                        .trustServiceId(671).uid(uid).purchaseToken(transactionId)
                        .userIp("127.0.0.1")
                        .build());

                if (!paymentResponse.isSuccess()) {
                    logger.warn("payment response {} is not success for payment {}", paymentResponse, payment);
                    continue;
                }

                Option<String> bindingResult =
                        Option.ofNullable(paymentResponse.getBindingResult()).map(Object::toString);
                if (bindingResult.isEmpty() || !bindingResult.get().equals("success")) {
                    logger.warn("card was not bind for request {}; bindingResult = ", payment, bindingResult);
                    continue;
                }
                Option<String> cardId =
                        Option.ofNullable(paymentResponse.getPaymethodId()).map(Object::toString);
                if (cardId.isEmpty()) {
                    logger.warn("no card id for request {}", payment);
                    continue;
                }

                ListF<GetBoundPaymentMethodsResponse> boundPaymentMethods =
                        Cf.list(balanceService.getBoundPaymentMethods(uid.getUid()));
                Option<GetBoundPaymentMethodsResponse> boundCardO =
                        boundPaymentMethods.find(x -> x.getPaymentMethodId().equals(cardId.get()));
                if (boundCardO.isEmpty()) {
                    logger.warn("card {} is no longer bound to user {}", cardId, uid);
                    continue;
                }
                GetBoundPaymentMethodsResponse boundCard = boundCardO.get();
                if (boundCard.isExpired()) {
                    logger.warn("card {} is expired for user {}", boundCard, uid);
                    continue;
                }

                CardPurpose purpose;
                ListF<CardEntity> cards = cardDao.findCardsByUid(uid);
                if (cards.find(x -> x.getExternalId().equals(boundCard.getPaymentMethodId())).isPresent()) {
                    logger.info("card {} already bind to user {}");
                    return;
                }

                if (cards.find(x -> x.getPurpose().equals(CardPurpose.B2B_PRIMARY)).isEmpty()) {
                    purpose = CardPurpose.B2B_PRIMARY;
                } else {
                    purpose = CardPurpose.B2B;
                }
                logger.info("binding card {} to user {} due the payment {} with purpose",
                        boundCard, uid, payment, purpose);
                cardDao.insertOrUpdateStatus(CardDao.InsertData.builder()
                        .uid(uid)
                        .purpose(purpose)
                        .externalId(boundCard.getPaymentMethodId())
                        .status(CardStatus.ACTIVE).build());

            } catch (Exception ex) {
                logger.error("unexpected error due processing payment {} : {}", payment, ex);
            }
        }
    }

}
