package ru.yandex.chemodan.app.psbilling.core.admin;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.admin.task.InappMigrationTask;
import ru.yandex.chemodan.app.psbilling.core.billing.users.PaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.billing.users.UserBillingService;
import ru.yandex.chemodan.app.psbilling.core.dao.admin.InappMigrationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.InappStore;
import ru.yandex.chemodan.app.psbilling.core.entities.admin.InappMigration;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserInfoService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.GetPaymentMethodRequest;
import ru.yandex.chemodan.trust.client.responses.PaymentMethod;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class InappMigrationService {
    @VisibleForTesting
    static final String INAPP_MIGRATION_SCHEDULED_MAIL_KEY = "inapp_migration_scheduled";
    @VisibleForTesting
    static final String INAPP_MIGRATION_FAILED_MAIL_KEY = "inapp_migration_failed";
    @VisibleForTesting
    static final String INAPP_MIGRATION_SUCCEED_MAIL_KEY = "inapp_migration_succeed";
    private static final Logger logger = LoggerFactory.getLogger(InappMigrationService.class);
    private static final MapF<String, MigrationProducts> PRODUCTS_MAPPING = Cf.hashMap();

    static {
    /*
    select inapp_up.code,
    web_up.code
    from user_products inapp_up
    left join (select up.code, up.code_family from product_lines
            join user_products_to_product_lines uptpl on product_lines.id = uptpl.product_line_id
            join user_products up on uptpl.user_product_id = up.id
            where description = 'web default line') web_up on regexp_replace(inapp_up.code_family, '_inapp.*', '') =
            web_up.code_family
    where inapp_up.billing_type like 'inap%'
    order by web_up.code nulls first;
    */
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium2000_introductory_inapp_google", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_light_v2_inapp_apple", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium2000_inapp_apple", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_light_inapp_apple_for_disk", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_light_inapp_google", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium2000_inapp_apple_for_disk", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_light_inapp_apple", null);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium1000_inapp_google", MigrationProducts.PREMIUM_1000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium1000_inapp_apple_for_disk", MigrationProducts.PREMIUM_1000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium1000_inapp_apple", MigrationProducts.PREMIUM_1000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium1000_introductory_inapp_google", MigrationProducts.PREMIUM_1000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium1000_inapp_apple_for_disk_test", MigrationProducts.PREMIUM_1000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium3000_inapp_apple_for_disk_test", MigrationProducts.PREMIUM_3000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium3000_inapp_google", MigrationProducts.PREMIUM_3000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium3000_inapp_apple", MigrationProducts.PREMIUM_3000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium3000_inapp_apple_for_disk", MigrationProducts.PREMIUM_3000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium3000_introductory_inapp_google", MigrationProducts.PREMIUM_3000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium5000_inapp_apple_for_disk", MigrationProducts.PREMIUM_5000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium5000_inapp_apple", MigrationProducts.PREMIUM_5000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium5000_introductory_inapp_google", MigrationProducts.PREMIUM_5000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium5000_inapp_google", MigrationProducts.PREMIUM_5000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_premium5000_inapp_apple_for_disk_test", MigrationProducts.PREMIUM_5000);
        PRODUCTS_MAPPING.put("mail_pro_b2c_standard100_inapp_apple", MigrationProducts.STANDARD_100);
        PRODUCTS_MAPPING.put("mail_pro_b2c_standard100_inapp_apple_for_disk", MigrationProducts.STANDARD_100);
        PRODUCTS_MAPPING.put("mail_pro_b2c_standard100_inapp_google", MigrationProducts.STANDARD_100);
        PRODUCTS_MAPPING.put("mail_pro_b2c_standard100_introductory_inapp_google", MigrationProducts.STANDARD_100);
        PRODUCTS_MAPPING.put("mail_pro_b2c_standard100_inapp_apple_for_disk_test", MigrationProducts.STANDARD_100);
    }


    private final UserBillingService userBillingService;
    private final UserProductManager userProductManager;
    private final OrderDao orderDao;
    private final InappMigrationDao inappMigrationDao;
    private final TrustClient trustClient;
    private final UserInfoService userInfoService;
    private final TaskScheduler taskScheduler;
    private final UserServiceManager userServiceManager;

    private final DynamicProperty<ListF<String>> availableRegions =
            new DynamicProperty<>("ps-billing.inapp_migration.available_regions", Cf.list("225"));

    private final DynamicProperty<Integer> migrationDelayMinutes =
            new DynamicProperty<>("ps-billing.inapp_migration.delay_minutes", 60 * 24 * 3); //3 days

    private final DynamicProperty<Integer> migrationBatchSize =
            new DynamicProperty<>("ps-billing.inapp_migration.batch_size", 100);

    private final DynamicProperty<Instant> migrationMinHoldDate =
            new DynamicProperty<>("ps-billing.inapp_migration.min_hold_date", new DateTime(
                    2022, 3, 10, 0, 0).toInstant());
    private final DynamicProperty<Instant> minOrderDateForDiscount =
            new DynamicProperty<>("ps-billing.inapp_migration.min_order_date_for_discount", new DateTime(
                    2022, 3, 9, 0, 0).toInstant());
    private DynamicProperty<Instant> migrationMaxPurchaseDate =
            new DynamicProperty<>("ps-billing.inapp_migration.max_purchase_date", new DateTime(
                    2022, 3, 23, 0, 0).toInstant());
    private DynamicProperty<Instant> maxOrderDateForDiscount =
            new DynamicProperty<>("ps-billing.inapp_migration.max_order_date_for_discount", new DateTime(
                    2022, 3, 22, 0, 0).toInstant());


    public void loadMigrationCandidates(boolean dryRun) {
        ListF<Order> orders = orderDao.findInappOrderForMigration(migrationMinHoldDate.get(),
                migrationMaxPurchaseDate.get());
        Set<String> uids = new HashSet<>();
        for (Order order : orders) {
            if (!uids.add(order.getUid())) {
                logger.info("uid {} already processed", order.getUid());
                continue;
            }
            if (uids.size() >= migrationBatchSize.get()) {
                logger.info("batch size limit reached. Stop");
                return;
            }
            UserProductPeriod currentPeriod = userProductManager.findPrice(order.getUserProductPriceId()).getPeriod();
            PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));
            InappMigration migration = inappMigrationDao.create(uid,
                    order.getId(),
                    MigrationStatus.INIT.name(),
                    currentPeriod.getCode());
            try {
                Option<UserProductPeriod> targetProductPeriod = findCorrespondingProduct(currentPeriod,
                        order.getCreatedAt());
                if (!targetProductPeriod.isPresent()) {
                    logger.error("Fail to find corresponding product period for order {} and period {}", order,
                            currentPeriod);
                    inappMigrationDao.setStatus(migration.getId(), MigrationStatus.ERROR_NO_CORRESPONDING_PRODUCTS);
                    continue;
                }
                UserProduct targetUserProduct = targetProductPeriod.get().getUserProduct();
                inappMigrationDao.setTargetProductPeriodCode(migration.getId(), targetProductPeriod.get().getCode());
                logger.info("found corresponding period for order {}. {} = {}",
                        order.getId(), currentPeriod.getCode(), targetProductPeriod);

                Integer trustServiceId = targetUserProduct.getTrustServiceId().get();

                Option<String> regionId = userInfoService.findOrCreateUserInfo(uid).getRegionId();
                if (regionId.isEmpty()) {
                    logger.info("Skip. No region id");
                    inappMigrationDao.setStatus(migration.getId(), MigrationStatus.ERROR_NO_REGION);
                    continue;
                } else if (!availableRegions.get().containsTs(regionId.get())) {
                    logger.info("Skip region id is {}, but available only {}", regionId.get(), availableRegions.get());
                    inappMigrationDao.setStatus(migration.getId(),
                            MigrationStatus.ERROR_NOT_AVAILABLE_REGION + " " + regionId.get());
                    continue;
                }

                List<PaymentMethod> availablePaymentMethods = getCardPaymentMethods(uid, trustServiceId);
                if (availablePaymentMethods.isEmpty()) {
                    logger.info("{} Skipped. No available payment methods", order);
                    inappMigrationDao.setStatus(migration.getId(), MigrationStatus.ERROR_NO_PAYMENT_METHODS);
                    continue;
                }
                if (!dryRun) {
                    taskScheduler.schedule(new InappMigrationTask(migration.getId()),
                            Instant.now().plus(Duration.standardMinutes(migrationDelayMinutes.get())));
                    logger.info("task scheduled");
                }
                inappMigrationDao.setStatus(migration.getId(), MigrationStatus.SCHEDULED + (dryRun ? " dryRun" : ""));
                if (!dryRun) {
                    taskScheduler.scheduleLocalizedEmailTask(INAPP_MIGRATION_SCHEDULED_MAIL_KEY,
                            MailContext.builder()
                                    .to(uid)
                                    .userProductId(Option.of(targetUserProduct.getId()))
                                    .inappStore(InappStore.fromBillingType(currentPeriod.getUserProduct().getBillingType()))
                                    .build());
                    logger.info("mail sent");
                }
            } catch (Exception e) {
                logger.error(e);
                inappMigrationDao.setStatus(migration.getId(), MigrationStatus.UNEXPECTED_ERROR_LOAD + e.getMessage());
            }
        }
    }

    public void subscribe(UUID migrationId) {
        subscribe(inappMigrationDao.findById(migrationId));
    }

    public void subscribe(InappMigration migration) {
        if (!migration.getStatus().equals(MigrationStatus.SCHEDULED.name())) {
            logger.info("Migration is not scheduled. Skip");
            return;
        }
        logger.info("Init resubscribe for {}", migration);
        inappMigrationDao.setStatus(migration.getId(), MigrationStatus.SUBSCRIPTION_STARTED);
        if (!orderDao.findById(migration.getOrderId()).getStatus().equals(OrderStatus.ON_HOLD)) {
            logger.info("canceled, not on hold");
            inappMigrationDao.setStatus(migration.getId(), MigrationStatus.SUBSCRIPTION_CANCELED_NOT_ON_HOLD);
            return;
        }
        boolean haveActiveServices = userServiceManager.findEnabled(migration.getUid().toString(), Option.empty())
                .filter(userService -> userService.getLastPaymentOrderId().isNotEmpty())
                .isNotEmpty();
        if (haveActiveServices) {
            logger.info("exists active subscription");
            inappMigrationDao.setStatus(migration.getId(),
                    MigrationStatus.SUBSCRIPTION_CANCELED_ACTIVE_SUBSCRIPTION_APPEARS);
            return;
        }
        try {
            UserProductPeriod targetProductPeriod = migration.getTargetProductPeriodCode()
                    .map(userProductManager::findPeriodByCode)
                    .orElseThrow(() -> new IllegalStateException("no target period"));
            UserProduct targetUserProduct = targetProductPeriod.getUserProduct();
            PassportUid uid = migration.getUid();

            Integer trustServiceId = targetProductPeriod.getUserProduct().getTrustServiceId().get();

            List<PaymentMethod> cardPaymentMethods = getCardPaymentMethods(uid, trustServiceId);
            UserProduct currentUserProduct = userProductManager.findPeriodByCode(
                    migration.getCurrentProductPeriodCode()).getUserProduct();
            Option<InappStore> inappStore = InappStore.fromBillingType(currentUserProduct.getBillingType());
            if (cardPaymentMethods.isEmpty()) {
                logger.info("no payment methods available");
                inappMigrationDao.setStatus(migration.getId(), MigrationStatus.SUBSCRIPTION_FAILED_NO_PAYMENT_METHODS);
                scheduleMigrationFailedMail(targetUserProduct, uid, inappStore);
                return;
            }
            boolean subscribed = false;
            for (PaymentMethod paymentMethod : cardPaymentMethods) {
                if (subscribed) {
                    return;
                }
                try {
                    PaymentInfo paymentInfo = userBillingService.initPayment(uid, null, "ru", Option.empty(),
                            Option.empty(),
                            targetProductPeriod.getCode(), Option.empty(), Option.empty(), true,
                            Option.empty(), paymentMethod.getId(), false, true, true);

                    String newOrderTrustId = paymentInfo.getOrder().getTrustOrderId();
                    Order newOrder;
                    int sleepTime = 5000;
                    while (true) {
                        if (sleepTime > 5 * 1000 * 60 * 30) { //not more than 30 minutes
                            logger.error("order check timeout");
                            inappMigrationDao.setStatus(migration.getId(), MigrationStatus.ORDER_CHECK_TIMEOUT);
                            return;
                        }
                        try {
                            userBillingService.checkOrder(newOrderTrustId);
                            newOrder = orderDao.findByTrustOrderId(newOrderTrustId).get();
                            if (newOrder.getStatus() != OrderStatus.INIT) {
                                break;
                            }
                            Thread.sleep(sleepTime);
                            sleepTime += sleepTime;
                        } catch (Exception e) {
                            logger.error("Error while checking order. Retry");
                        }
                    }
                    if (newOrder.getStatus() == OrderStatus.ERROR) {
                        logger.info("Subscription failed for paymentMethod {}", paymentMethod.getId());
                        continue;
                    }
                    subscribed = true;
                    logger.info("migrated. new order trustId is {} ", newOrderTrustId);
                    inappMigrationDao.setStatus(migration.getId(), MigrationStatus.MIGRATED);
                    taskScheduler.scheduleLocalizedEmailTask(INAPP_MIGRATION_SUCCEED_MAIL_KEY,
                            MailContext.builder()
                                    .to(uid)
                                    .userServiceId(newOrder.getUserServiceId().map(UUID::toString))
                                    .userProductPriceId(Option.of(newOrder.getUserProductPriceId()))
                                    .cardName(Option.of(paymentMethod.getFormattedName()))
                                    .inappStore(inappStore)
                                    .build());
                    logger.info("mail sent");
                    return;
                } catch (Exception e) {
                    logger.error("unexpected error while subscribing", e);
                }
            }
            inappMigrationDao.setStatus(migration.getId(),
                    MigrationStatus.SUBSCRIPTION_FAILED_ALL_PAYMENT_METHODS_FAILED);
            scheduleMigrationFailedMail(targetUserProduct, uid, inappStore);
        } catch (Exception e) {
            logger.error(e);
            inappMigrationDao.setStatus(migration.getId(),
                    MigrationStatus.UNEXPECTED_ERROR_SUBSCRIBE + " " + e.getMessage());
        }
    }

    private void scheduleMigrationFailedMail(UserProduct targetUserProduct, PassportUid uid,
                                             Option<InappStore> inappStore) {
        taskScheduler.scheduleLocalizedEmailTask(INAPP_MIGRATION_FAILED_MAIL_KEY,
                MailContext.builder()
                        .to(uid)
                        .userProductId(Option.of(targetUserProduct.getId()))
                        .inappStore(inappStore)
                        .build());
    }

    private List<PaymentMethod> getCardPaymentMethods(PassportUid uid, Integer trustServiceId) {
        GetPaymentMethodRequest request = new GetPaymentMethodRequest(trustServiceId, uid, null, null);
        return trustClient.getPaymentMethods(request).getPaymentMethods()
                .stream()
                .filter(method -> !method.isExpired())
                .filter(method -> method.getPaymentMethod().equals("card"))
                .collect(Collectors.toList());
    }

    private Option<UserProductPeriod> findCorrespondingProduct(UserProductPeriod currentPeriod, Instant createdAt) {
        UserProduct currentProduct = currentPeriod.getUserProduct();
        Option<String> correspondingCode;
        if (createdAt.isAfter(minOrderDateForDiscount.get()) && createdAt.isBefore(maxOrderDateForDiscount.get())) {
            correspondingCode = PRODUCTS_MAPPING.getO(currentProduct.getCode()).map(product -> product.discountCode);
        } else {
            correspondingCode = PRODUCTS_MAPPING.getO(currentProduct.getCode()).map(product -> product.regularCode);
        }

        return correspondingCode
                .flatMapO(userProductManager::findByCodeO)
                .flatMap(UserProduct::getProductPeriods)
                .filter(period -> period.getPeriod().equals(currentPeriod.getPeriod()))
                .firstO();
    }

    @VisibleForTesting
    void setMigrationMaxPurchaseDate(DynamicProperty<Instant> migrationMaxPurchaseDate) {
        this.migrationMaxPurchaseDate = migrationMaxPurchaseDate;
    }

    @VisibleForTesting
    void setMaxOrderDateForDiscount(DynamicProperty<Instant> maxOrderDateForDiscount) {
        this.maxOrderDateForDiscount = maxOrderDateForDiscount;
    }

    public enum MigrationStatus {
        INIT,
        ERROR_NO_CORRESPONDING_PRODUCTS,
        ERROR_NO_REGION,
        ERROR_NOT_AVAILABLE_REGION,
        ERROR_NO_PAYMENT_METHODS,
        UNEXPECTED_ERROR_LOAD,
        SCHEDULED,
        SUBSCRIPTION_STARTED,
        SUBSCRIPTION_CANCELED_NOT_ON_HOLD,
        SUBSCRIPTION_CANCELED_ACTIVE_SUBSCRIPTION_APPEARS,
        SUBSCRIPTION_FAILED_NO_PAYMENT_METHODS,
        MIGRATED,
        SUBSCRIPTION_FAILED_ALL_PAYMENT_METHODS_FAILED,
        ORDER_CHECK_TIMEOUT,
        UNEXPECTED_ERROR_SUBSCRIBE
    }


    private enum MigrationProducts {
        STANDARD_100("mail_pro_b2c_standard100_v20210610", "mail_pro_b2c_standard100_start_discount_50_1m"),
        PREMIUM_1000("mail_pro_b2c_premium1000_v20210610", "mail_pro_b2c_premium1000_start_discount_50_1m"),
        PREMIUM_3000("mail_pro_b2c_premium3000_v20210610", "mail_pro_b2c_premium3000_start_discount_50_1m"),
        //no discount for 5TB so use same code,
        PREMIUM_5000("mail_pro_b2c_premium5000_v20210610", "mail_pro_b2c_premium5000_v20210610");

        String regularCode;
        String discountCode;
        MigrationProducts(String regularCode, String discountCode) {
            this.regularCode = regularCode;
            this.discountCode = discountCode;
        }
    }
}
