package ru.yandex.chemodan.app.psbilling.core.admin.task;

import javax.annotation.Nullable;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.intellij.lang.annotations.Language;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.http.HttpStatus;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.TrustException;
import ru.yandex.chemodan.trust.client.requests.GetPaymentMethodRequest;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.SupplementRequest;
import ru.yandex.chemodan.trust.client.responses.PaymentMethod;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class SupplementNonRusTask extends OnetimeTaskSupport<SupplementNonRusTask.Parameters> {
    private final DynamicProperty<Integer> timeToShiftDays =
            new DynamicProperty<>("ps-billing.supplement.daysToShift", 30);
    private final Logger logger = LoggerFactory.getLogger(SupplementNonRusTask.class);

    private TrustClient trustClient;
    private OrderProcessorFacade orderProcessorFacade;
    private JdbcTemplate3 jdbcTemplate;

    public SupplementNonRusTask(TrustClient trustClient, OrderProcessorFacade orderProcessorFacade,
                                JdbcTemplate3 jdbcTemplate) {
        super(SupplementNonRusTask.Parameters.class);
        this.trustClient = trustClient;
        this.orderProcessorFacade = orderProcessorFacade;
        this.jdbcTemplate = jdbcTemplate;
    }

    public SupplementNonRusTask(PassportUid uid, Integer trustServiceId, String orderId, boolean dryRun) {
        super(new SupplementNonRusTask.Parameters(uid, orderId, trustServiceId, dryRun));
    }

    @Override
    protected void execute(SupplementNonRusTask.Parameters parameters, ExecutionContext context) {
        PassportUid uid = parameters.uid;
        String orderId = parameters.orderId;
        Integer trustServiceId = parameters.trustServiceId;
        boolean dryRun = getParametersTyped().dryRun;
        boolean isMpfs = parameters.isMpfs();

        execute(uid, orderId, trustServiceId, dryRun, isMpfs);
    }

    public void execute(PassportUid uid, String orderId, Integer trustServiceId, boolean dryRun, boolean isMpfs) {
        String result = process(uid, orderId, trustServiceId, dryRun, isMpfs);
        logger.info("result is {}", result);
        @Language("SQL") String sql = "" +
                "insert into supplemented_users (uid, trust_order_id, status, created_at)\n" +
                "values (?,?,?, now())\n" +
                "returning *";
        jdbcTemplate.query(sql, r -> null, uid, orderId, result);
    }

    public String process(PassportUid uid, String orderId, Integer trustServiceId, boolean dryRun, boolean isMpfs) {
        Duration timeToShift = Duration.standardDays(timeToShiftDays.get());

        String postfix = String.format("uid %s; order %s; trustServiceId %s;", uid, orderId, trustServiceId);
        logger.info("processing; {}", postfix);

        try {
            SubscriptionResponse subscription = trustClient.getSubscription(OrderRequest.builder()
                    .uid(uid)
                    .trustServiceId(trustServiceId)
                    .orderId(orderId)
                    .build());

            Instant subscriptionUntil = subscription.getSubscriptionUntil();
            Instant newCheckDate = subscriptionUntil.plus(timeToShift);

            Option<Object> nextChargeMethodO = subscription.getOtherAttributes().getO("next_charge_payment_method");
            if (nextChargeMethodO.isEmpty()) {
                logger.warn("no next payment method. {}", postfix);
                return "no next payment method";
            }
            String nextChargeMethod = nextChargeMethodO.get().toString();

            if (subscriptionUntil.isAfter(Instant.now().plus(timeToShift))) {
                logger.info("duration is already after target date; {}", postfix);
                return "already ok";
            }

            GetPaymentMethodRequest request = new GetPaymentMethodRequest(trustServiceId, uid, null, null);
            ListF<PaymentMethod> paymentMethods = Cf.toList(trustClient.getPaymentMethods(request).getPaymentMethods());
            Option<PaymentMethod> paymentMethodO = paymentMethods.find(x -> x.getId().equals(nextChargeMethod));

            if (paymentMethodO.isEmpty()) {
                logger.info("corresponding card not found; {}", postfix);
                return "no card";
            }
            PaymentMethod paymentMethod = paymentMethodO.get();

            Option<Object> cardCountryO = paymentMethod.getOtherAttributes().getO("card_country");
            if (cardCountryO.isEmpty()) {
                logger.warn("no card country for payment method {}. {}", paymentMethod, postfix);
                return "no card country";
            }
            String cardCountry = cardCountryO.get().toString();
            logger.info("card country is {}; {}", cardCountry, postfix);
            if (cardCountry.equals("RUS")) {
                return "wrong card country";
            }

            logger.info("going to supplement subscription {} with new date newCheckDate {} cause card country is {}; " +
                            "{}",
                    subscription, newCheckDate, cardCountry, postfix);
            if (dryRun) {
                logger.info("dryRun. Exiting; {}", postfix);
                return "dry run. success";
            }
            SubscriptionResponse response = trustClient.supplementSubscription(SupplementRequest.builder()
                    .trustServiceId(trustServiceId)
                    .uid(uid)
                    .orderId(orderId)
                    .expectedSubsUntilTs(subscriptionUntil)
                    .supplementUntilTs(newCheckDate)
                    .build());

            if (response.getSubscriptionUntil().equals(newCheckDate)) {
                logger.info("success. response: {}; {}", response, postfix);
                if (!isMpfs) {
                    logger.info("going to check order {}; {}", orderId, postfix);
                    orderProcessorFacade.processByTrustOrderId(orderId);
                }
                return "success";
            } else {
                logger.error("NOT success for uid {} and order {}", uid, orderId);
                return "error";
            }
        } catch (TrustException ex) {
            if (ex.getHttpCode().isMatch(c -> c == HttpStatus.TOO_MANY_REQUESTS)) {
                logger.error("429 Trust error. will try later; {}", ex, postfix);
                throw ex;
            }
            logger.error("unexpected trust error n: {}; {}", ex, postfix);
            return "unexpected error";
        } catch (Exception ex) {
            logger.error("unexpected error during getting subscription: {}; {}", ex, postfix);
            return "unexpected error";
        }
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(5), Integer.MAX_VALUE);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.REGULAR;
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return SupplementNonRusTask.Parameters.Converter.class;
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final PassportUid uid;
        private final String orderId;
        private final Integer trustServiceId;
        private final boolean dryRun;

        private boolean isMpfs() {
            return trustServiceId != 690;
        }

        public static class Converter implements
                ActiveUniqueIdentifierConverter<SupplementNonRusTask.Parameters, SupplementNonRusTask.Parameters> {
            @Override
            public Class<SupplementNonRusTask.Parameters> getActiveUniqueIdentifierClass() {
                return SupplementNonRusTask.Parameters.class;
            }

            @Override
            public SupplementNonRusTask.Parameters convert(SupplementNonRusTask.Parameters parameters) {
                return parameters;
            }
        }
    }
}
