package ru.yandex.chemodan.app.psbilling.core.balance;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.joda.time.LocalDate;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.config.BalanceOfferConfig;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.balanceclient.BalanceClient;
import ru.yandex.chemodan.balanceclient.exception.BalanceException;
import ru.yandex.chemodan.balanceclient.model.request.CheckBindingRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateClientRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateInvoiceRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateOfferRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreatePersonRequest;
import ru.yandex.chemodan.balanceclient.model.request.CreateRequest2Item;
import ru.yandex.chemodan.balanceclient.model.request.FindClientRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetBoundPaymentMethodsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetCardBindingURLRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetClientActsRequest;
import ru.yandex.chemodan.balanceclient.model.request.GetClientContractsRequest;
import ru.yandex.chemodan.balanceclient.model.request.PayRequestRequest;
import ru.yandex.chemodan.balanceclient.model.response.CheckBindingResponse;
import ru.yandex.chemodan.balanceclient.model.response.CheckRequestPaymentResponse;
import ru.yandex.chemodan.balanceclient.model.response.ClientActInfo;
import ru.yandex.chemodan.balanceclient.model.response.ClientPassportInfo;
import ru.yandex.chemodan.balanceclient.model.response.CreateRequest2Response;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetBoundPaymentMethodsResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetCardBindingURLResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetClientPersonsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetOrdersInfoResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetPartnerBalanceContractResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.PayRequestResponse;
import ru.yandex.chemodan.balanceclient.model.response.PaymentMethodDetails;
import ru.yandex.chemodan.balanceclient.model.response.PersonPaymentMethodDetails;
import ru.yandex.chemodan.balanceclient.model.response.RequestPaymentMethod;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @see <a href="https://wiki.yandex-team.ru/users/isupov/Eda-Jurlica/"/>
 */
@AllArgsConstructor
public class BalanceService {
    private static final Logger logger = LoggerFactory.getLogger(BalanceService.class);
    public static final Long COUNTRY_RUSSIA = 225L;
    private static final String CURRENCY_RUB = "RUB";
    private static final String FIRM_ID_YANDEX = "1";
    public static final String PAYMENT_METHOD_BANK = "bank";
    public static final String TRUST_PAYMENT_FORM_METHOD_ID = "trust_web_page";
    public static final String CARD_PAYMENT_METHOD = "card";
    public static final String SUCCESS_PAYMENT_STATUS = "success";
    private final DynamicProperty<Boolean> GET_PAYERS_FROM_BALANCE = new DynamicProperty<>(
            "ps-billing.organization-payers-from-balance", true);

    private final BalanceClient balanceClient;
    private final Integer serviceId;
    private final BalanceOfferConfig offerConfig;
    private final RestTemplate balanceRestTemplate;
    private final String balanceUrl;
    private final String groupPaymentRequestNotificationUrlTemplate;
    private final String groupCardBindingNotificationUrlTemplate;
    private final String selfClientId;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final MapF<Integer, String> serviceTokens;


    public long createPaymentRequest(PassportUid uid, long clientId, String contractId, BigDecimal amount) {
        CreateRequest2Response request = balanceClient
                .createRequest(uid.getUid(), clientId, createRequest2Item(contractId, amount), false,
                        Option.of("charge_note"));
        logger.info("Created payment request2 : {}", request);
        return request.getRequestID();
    }

    public Option<RequestPaymentMethod> getRequestPaymentMethodForUnboundCard(PassportUid uid, String requestId,
                                                                              String contractId) {
        List<RequestPaymentMethod> paymentMethods =
                balanceClient.getRequestPaymentMethodsForContract(uid, requestId, contractId);
        return Option.x(paymentMethods.stream().filter(requestPaymentMethod ->
                CARD_PAYMENT_METHOD.equals(requestPaymentMethod.getPaymentMethodType()) &&
                        !requestPaymentMethod.getPaymentMethodId().isPresent()).findFirst());
    }

    public ListF<RequestPaymentMethod> getBoundCards(PassportUid uid, String requestId, String contractId) {
        List<RequestPaymentMethod> paymentMethods =
                balanceClient.getRequestPaymentMethodsForContract(uid, requestId, contractId);
        return Cf.toList(paymentMethods).filter(requestPaymentMethod ->
                CARD_PAYMENT_METHOD.equals(requestPaymentMethod.getPaymentMethodType()) &&
                        requestPaymentMethod.getPaymentMethodId().isPresent());
    }


    public PersonPaymentMethodDetails getRequestPaymentChoices(PassportUid uid, GetClientContractsResponseItem contract,
                                                               long paymentRequestId) {
        List<PersonPaymentMethodDetails> paymentMethodDetails =
                balanceClient.getPaymentMethods(uid.getUid(), contract.getId(), paymentRequestId)
                        .getPersonPaymentMethodDetails();

        List<PersonPaymentMethodDetails> contractPaymentMethods = paymentMethodDetails.stream()
                .filter(p -> p.getPerson() != null && p.getContract() != null &&
                        Objects.equals(p.getPerson().getId(), contract.getPersonId())
                        && Objects.equals(p.getContract().getId(), contract.getId())
                ).collect(Collectors.toList());

        if (contractPaymentMethods.isEmpty()) {
            throw new IllegalStateException("Payment methods not found");
        }
        if (contractPaymentMethods.size() > 1) {
            logger.warn("Found several payment methods for contract {} and person {}: {}",
                    contract.getId(), contract.getPersonId(), contractPaymentMethods);
        }

        return contractPaymentMethods.get(0);
    }

    public String getInvoiceMdsUrl(long invoiceId) {
        URI url = UriBuilder.cons(balanceUrl).appendPath("/documents/invoices").appendPath(String.valueOf(invoiceId))
                .build();
        ResponseEntity<String> response = balanceRestTemplate.getForEntity(url, String.class);
        try {
            MdsLinkResponse mdsLinkResponse = objectMapper.readValue(response.getBody(), MdsLinkResponse.class);
            return mdsLinkResponse.url;
        } catch (IOException e) {
            logger.error("Failed to parse response from balance invoice pdf generator: {}", response.getBody());
            throw new RuntimeException("Failed to parse response from balance invoice pdf generator", e);
        }
    }

    public long createInvoice(PassportUid uid, GetClientContractsResponseItem contract, long paymentRequestId) {
        PersonPaymentMethodDetails requestPaymentChoices =
                getRequestPaymentChoices(uid, contract, paymentRequestId);

        List<PaymentMethodDetails> bankPaymentMethod = requestPaymentChoices.getPaymentMethodDetails().stream()
                .filter(m -> PAYMENT_METHOD_BANK.equals(m.getPaymentMethodCode()))
                .collect(Collectors.toList());
        if (bankPaymentMethod.isEmpty()) {
            throw new IllegalStateException("Not found " + PAYMENT_METHOD_BANK + " payment method");
        }
        if (bankPaymentMethod.size() > 1) {
            logger.warn("Found several " + PAYMENT_METHOD_BANK + " payment methods for contract {} and person {}: {}",
                    contract.getId(), contract.getPersonId(), bankPaymentMethod);
        }

        PaymentMethodDetails paymentMethod = bankPaymentMethod.get(0);
        return balanceClient.createInvoice(new CreateInvoiceRequest(
                uid.getUid(), paymentRequestId, paymentMethod.getId(), contract.getPersonId(), contract.getId()));
    }

    private GetOrdersInfoResponseItem findOrdersInfoForContract(String contractId) {
        GetOrdersInfoResponseItem[] ordersInfo = balanceClient.getOrdersInfo(contractId);
        if (ordersInfo.length == 0) {
            throw new IllegalStateException("Unable to find orders info for contract " + contractId);
        }
        if (ordersInfo.length > 1) {
            logger.warn("Found several orders info for contract {} : {}", contractId, ordersInfo);
        }
        return ordersInfo[0];
    }

    public Option<GetClientContractsResponseItem> getActiveContract(PassportUid uid) {
        return findClient(uid).map(this::getActiveContract).getOrElse(Option.empty());
    }

    public Option<GetClientContractsResponseItem> getActiveContract(long clientId) {
        GetClientContractsRequest request = new GetClientContractsRequest().withClientId(clientId);

        return filterActiveContract(Cf.x(balanceClient.getClientContracts(request)));
    }

    private Option<GetClientContractsResponseItem> filterActiveContract(ListF<GetClientContractsResponseItem> contracts) {
        ListF<GetClientContractsResponseItem> activeContracts = contracts
                .filter(c -> c.getServices().contains(serviceId))
                .filter(GetClientContractsResponseItem::isActive);
        if (activeContracts.size() > 1) {
            logger.warn("found more than one active contracts: {}", activeContracts);
        }
        return activeContracts.firstO();
    }

    public ListF<GetClientContractsResponseItem> getClientContractsWithFinished(long clientId) {
        GetClientContractsRequest request = new GetClientContractsRequest()
                .withAddFinished(true)
                .withClientId(clientId);
        ListF<GetClientContractsResponseItem> result = Cf.x(balanceClient.getClientContracts(request))
                .filter(c -> c.getServices().contains(serviceId))
                //нужны только договора с датой начала в прошлом или равной текущей. договора из будущего не смотрим
                .filter(c -> c.getStartDate() == null || !c.getStartDate().isAfter(LocalDate.now()));
        logger.info("Found contracts {}", result);

        return result;
    }

    public ListF<GetClientContractsResponseItem> getClientContracts(long clientId) {
        GetClientContractsRequest request = new GetClientContractsRequest().withClientId(clientId);
        GetClientContractsResponseItem[] contracts = balanceClient.getClientContracts(request);
        ListF<GetClientContractsResponseItem> result = Cf.x(contracts)
                .filter(c -> c.getServices().contains(serviceId))
                //нужны только договора с датой начала в прошлом или равной текущей. договора из будущего не смотрим
                .filter(c -> c.getStartDate() == null || !c.getStartDate().isAfter(LocalDate.now()));
        logger.info("Found contracts {}", result);
        return result;
    }

    public ListF<GetPartnerBalanceContractResponseItem> getContractsBalance(ListF<Long> contractIds) {
        return Cf.x(balanceClient.getPartnerBalance(serviceId, contractIds));
    }

    public MapF<Long, PaymentData> findPaymentData(Long clientId) {
        return Cf.x(balanceClient.getClientPersons(clientId))
                .toMap(GetClientPersonsResponseItem::getId, PaymentData::of)
                .filterValues(paymentData -> !(paymentData instanceof PaymentDataWithUnknownType));
    }

    public MapF<Long, PaymentData> findPaymentData(PassportUid uid) {
        Option<Long> clientId = findClient(uid);
        if (!clientId.isPresent()) {
            return Cf.map();
        }

        return findPaymentData(clientId.get());
    }

    public Option<Email> findActiveContractPaymentEmail(long clientId) {
        Option<GetClientContractsResponseItem> response = getActiveContract(clientId);

        if (response.isEmpty()) {
            logger.debug("No active contract found for client {}", clientId);
            return Option.empty();
        }
        long personId = response.get().getPersonId();

        Option<PaymentData> data = findPaymentData(clientId).getO(personId);

        if (data.isEmpty()) {
            logger.debug("No payment data found for client {} and person {}", clientId, personId);
            return Option.empty();
        }
        Option<Email> email = Option.ofNullable(data.get().getEmail())
                .map(Utils::parseFirstEmailFromBalance)
                .flatMapO(Email::parseSafe);

        if (email.isEmpty()) {
            logger.debug("No valid email found for client {} and person {}", clientId, personId);
            return Option.empty();
        }
        return email;
    }

    public Long findOrCreateClient(PassportUid uid, String name, String email, String phone) {
        Option<Long> clientIdO = findClient(uid);
        return clientIdO.orElseGet(() -> createClient(uid, name, email, phone));
    }

    public Long createPerson(PassportUid uid, Long clientId, PaymentData paymentData) {
        CreatePersonRequest createPersonRequest = paymentData.toCreatePersonRequest()
                .withOperatorUid(uid.getUid())
                .withClientId(clientId.toString())
                .withIsPartner(false);

        return balanceClient.createPerson(createPersonRequest);
    }

    public ListF<PassportUid> findGroupPayers(Group group) {
        Option<BalancePaymentInfo> paymentInfo = group.getPaymentInfoWithUidBackwardCompatibility();
        if (!paymentInfo.isPresent()) {
            return Cf.list();
        }

        if (!GET_PAYERS_FROM_BALANCE.get()) {
            return paymentInfo.map(BalancePaymentInfo::getPassportUid);
        }

        long clientId = paymentInfo.get().getClientId();
        List<ClientPassportInfo> clientRepresentativePassports =
                balanceClient.getClientRepresentativePassports(clientId, Option.empty());
        return Cf.x(clientRepresentativePassports).map(ClientPassportInfo::getUid).map(PassportUid::cons);
    }

    public long createOffer(PassportUid uid, long clientId, long personId) {
        CreateOfferRequest request = new CreateOfferRequest()
                .withOperatorUid(uid.getUid())
                .withClientId(String.valueOf(clientId))
                .withPersonId(String.valueOf(personId))
                .withCurrency(CURRENCY_RUB)
                .withServices(Collections.singleton(serviceId))
                .withFirmId(FIRM_ID_YANDEX)
                .withIsPersonalAccount(offerConfig.isPersonalAccount())
                .withPaymentType(CreateOfferRequest.PAYMENT_TYPE_POSTPAID)
                .withCountry(COUNTRY_RUSSIA.toString())
                .withPaymentTerm(offerConfig.getPaymentTerm());
        if (StringUtils.isNotEmpty(offerConfig.getManagerUid())) {
            request.withManagerUid(offerConfig.getManagerUid());
        }
        request.setMaxRetries(0);

        return balanceClient.createOffer(request).getId();
    }

    public PayRequestResponse createPayRequestByTrustForm(PassportUid uid, String requestId, String contractId,
                                                          String currency, String redirectUrl, Option<String> theme,
                                                          Option<String> title, Option<String> payload) {
        PayRequestRequest payRequest = new PayRequestRequest()
                .withOperatorUid(uid.getUid())
                .withRequestId(requestId)
                .withPaymentMethodId(TRUST_PAYMENT_FORM_METHOD_ID)
                .withCurrency(currency)
                .withContractId(contractId)
                .withRedirectUrl(redirectUrl)
                .addToPayload("css-theme", theme)
                .addToPayload("title", title)
                .addToPayload("notify-tvm-id", selfClientId)
                .withNotificationUrl(groupPaymentRequestNotificationUrlTemplate.replaceAll("!REQUEST_ID!", requestId));
        payload.ifPresent(payRequest::withPayload);

        return balanceClient.payRequest(payRequest);
    }

    public PayRequestResponse createAutomaticPayRequest(PassportUid uid, String requestId, String contractId,
                                                        String paymentId,
                                                        String currency) {
        PayRequestRequest payRequest = new PayRequestRequest()
                .withOperatorUid(uid.getUid())
                .withRequestId(requestId)
                .withPaymentMethodId(paymentId)
                .withCurrency(currency)
                .withContractId(contractId)
                .addToPayload("notify-tvm-id", selfClientId)
                .asRecurrentPayment()
                .withNotificationUrl(groupPaymentRequestNotificationUrlTemplate.replaceAll("!REQUEST_ID!", requestId));

        return balanceClient.payRequest(payRequest);
    }

    private Long createClient(PassportUid uid, String name, String email, String phone) {
        CreateClientRequest createClientRequest = new CreateClientRequest()
                .withServiceId(serviceId)
                .withEmail(email)
                .withName(name)
                .withPhone(phone);

        long client = balanceClient.createClient(uid.getUid(), createClientRequest);

        try {
            balanceClient.createUserClientAssociation(uid.getUid(), client, uid.getUid());
            return client;
        } catch (BalanceException e) {
            //try to find already associated client
            Option<Long> clientO = findClient(uid);
            return clientO.orElseThrow(() -> e);
        }
    }

    public Option<Long> findClient(PassportUid uid) {
        return findClientData(uid).map(FindClientResponseItem::getClientId);
    }

    public Option<FindClientResponseItem> findClientData(PassportUid uid) {
        List<FindClientResponseItem> clients = balanceClient.findClient(new FindClientRequest().withUid(uid.getUid()));
        if (clients.isEmpty()) {
            return Option.empty();
        }
        if (clients.size() > 1) {
            logger.warn(
                    "balanceClient.findClient returns more than 1 clients in search by uid. Will be used first one");
        }

        return Cf.x(clients).firstO();
    }

    public Integer getServiceId() {
        return serviceId;
    }

    public CheckRequestPaymentResponse checkRequestPayment(String requestId, Option<String> transactionId,
                                                           Long operatorUid) {
        return balanceClient.checkRequestPayment(serviceId, requestId, transactionId, operatorUid);
    }

    public ListF<ClientActInfo> getClientActs(Long clientId) {
        String token = serviceTokens.getOrThrow(serviceId,
                "Unable to find serviceToken for balanceService " + serviceId);
        return Cf.x(balanceClient.getClientActs(new GetClientActsRequest(clientId).withServiceToken(token)));
    }

    public GetBoundPaymentMethodsResponse[] getBoundPaymentMethods(Long uid) {
        return balanceClient.getBoundPaymentMethods(new GetBoundPaymentMethodsRequest(serviceId).withOperatorUid(uid));
    }

    public CheckBindingResponse checkBinding(Long operatorUid, String purchaseToken) {
        return balanceClient.checkBinding(new CheckBindingRequest().
                withOperatorUid(operatorUid).
                withServiceId(serviceId).
                withPurchaseToken(purchaseToken));
    }

    public GetCardBindingURLResponse getCardBindingURL(Long operatorUid, String currencyCode,
                                                       String bindingId, Option<String> theme,
                                                       Option<String> title) {
        GetCardBindingURLRequest request = new GetCardBindingURLRequest()
                .withCurrency(currencyCode)
                // Надо вернуть, когда notify ручка будет дёргаться балансом (см. https://st.yandex-team.ru/CHEMODAN-82878)
                //.withReturnPath(groupCardBindingNotificationUrlTemplate.replaceAll("!BINDING_ID!", bindingId))
                .withServiceID(serviceId)
                .withOperatorUid(operatorUid)
                .addToPayload("css-theme", theme)
                .addToPayload("title", title)
                .addToPayload("notify-tvm-id", selfClientId);

        return balanceClient.getCardBindingURL(request);
    }

    private CreateRequest2Item createRequest2Item(String contractId, BigDecimal amount) {
        GetOrdersInfoResponseItem ordersInfo = findOrdersInfoForContract(contractId);

        return new CreateRequest2Item()
                .withQty(amount)
                .withServiceId(ordersInfo.getServiceId())
                .withServiceOrderId(ordersInfo.getServiceOrderId());
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class MdsLinkResponse {
        @JsonProperty("mds_link")
        private String url;
    }
}
