package ru.yandex.chemodan.app.psbilling.core.billing;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.texts.PredefinedTankerKey;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;

import static ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation.DEFAULT_LANGUAGE;


@RequiredArgsConstructor
public class TrustDeveloperPayloadCreator {
    private final TextsManager textsManager;

    public Map<String, Object> createCreatePaymentPayload(boolean useTemplate, UserProduct product) {
        Map<String, Object> payload = new HashMap<>();

        if (useTemplate) {
            payload.put("template", "checkout_tuning");
        }

        blocksVisibilityAmount(payload, product);

        return payload;
    }

    public Map<String, Object> createCreateOrderPayload(
            UserProductPeriod period,
            Option<String> language,
            boolean disableTrustHeader
    ) {
        MapF<String, Object> payload = Cf.hashMap();

        title(language, payload);
        tariffName(period, language, payload);
        blocksVisibilityAmount(payload, period.getUserProduct());

        if (disableTrustHeader) {
            payload.put("blocks_visibility", Cf.map("header", false));
        }

        return payload;
    }

    private void title(Option<String> language, MapF<String, Object> payload) {
        textsManager.findPredefinedTankerTranslation(PredefinedTankerKey.Yandex360)
                .findLanguageTranslation(language.orElse(DEFAULT_LANGUAGE))
                .ifPresent(t -> payload.put("title", t));
    }

    private void tariffName(UserProductPeriod period, Option<String> language, MapF<String, Object> payload) {
        period.getUserProduct().getTitle()
                .flatMapO(t -> t.findLanguageTranslation(language.orElse(DEFAULT_LANGUAGE)))
                .ifPresent(p -> {
                    String title = localizedUserProductPeriodTitle(period, language);
                    payload.put("tariff_name", String.format("%s %s", p,title));
                });
    }

    private void blocksVisibilityAmount(Map<String, Object> payload, UserProduct product) {
        product.getTrialDefinition()
                .filter(def -> BigDecimal.ZERO.compareTo(def.getPrice()) == 0)
                .ifPresent(def -> payload.put("blocks_visibility", Cf.map("amount", false)));
    }

    private String localizedUserProductPeriodTitle(UserProductPeriod upp, Option<String> language) {
        if (upp.hasStartPeriod()) {
            return "";
        }
        return textsManager.localizedCustomPeriodTitle(upp.getPeriod(), language);
    }

}
