package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks.GroupAutoBillingTask;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class AutoPayManager extends BasePayManager {
    private static final Logger logger = LoggerFactory.getLogger(AutoPayManager.class);

    public AutoPayManager(ClientBalanceDao clientBalanceDao, Settings settings,
                          GroupTrustPaymentRequestDao groupTrustPaymentRequestDao,
                          ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService groupBillingService,
                          GroupDao groupDao, ClientBalanceCalculator clientBalanceCalculator, CardDao cardDao,
                          GroupServiceDao groupServiceDao, GroupProductManager groupProductManager,
                          BalanceService balanceService, TaskScheduler taskScheduler, FeatureFlags featureFlags,
                          DirectoryService directoryService) {
        super(clientBalanceDao, settings, groupTrustPaymentRequestDao, groupBillingService, groupDao,
                clientBalanceCalculator, cardDao, groupServiceDao, groupProductManager, balanceService, taskScheduler,
                featureFlags, directoryService);
    }

    @Override
    protected Duration getMinRetryInterval() {
        return settings.getAutoPayMinRetryIntervalMinutes();
    }

    public void scheduleAutoPayment() {
        Instant balanceVoidBefore = Instant.now().plus(settings.getAutoPayLeadTime());

        BatchFetchingUtils.<Tuple2<Long, Currency>, Tuple2<Long, Currency>>fetchAndProcessBatchedEntities(
                (batchSize, lastData) ->
                        clientBalanceDao.findClientsForAutoPay(balanceVoidBefore, batchSize, lastData),
                Function.identityF(),
                clientAndCurrency ->
                        taskScheduler.schedule(new GroupAutoBillingTask(clientAndCurrency._1, clientAndCurrency._2)),
                CLIENTS_FETCH_BATCH_SIZE,
                logger);
    }

    // charge regular payment for enabled services
    public void chargeRegularAutoPayment(Long clientId, Currency currency) {
        BalancePaymentInfo paymentInfo = getPaymentInfo(clientId);
        ListF<GroupTrustPaymentRequest> recentPayments = getRecentPayments(clientId);
        PassportUid uid = paymentInfo.getPassportUid();
        Option<CardEntity> cardO = cardDao.findB2BPrimary(uid);

        if (!validateExecutionIsNeeded(paymentInfo, currency, recentPayments, cardO)) {
            return;
        }

        CardEntity card = cardO.orElseThrow(() -> new IllegalStateException("Uid " + uid + " does not have a primary card in b2b"));
        Double nextChargeCoefficient = getNextChargeCoefficient(recentPayments, card.getId());

        ListF<GroupService> groupServices = groupServiceDao.findActiveGroupServicesByPaymentInfoClient(
                clientId, currency);
        Money chargeAmount = super.getChargeAmount(groupServices, currency, nextChargeCoefficient);

        if (chargeAmount.getAmount().compareTo(BigDecimal.ZERO) == 0) {
            logger.info("no charge required cause total price of all active services is zero for client {} and " +
                    "currency {}", clientId, currency);
            return;
        }

        groupBillingService.chargePayment(uid, clientId, chargeAmount, card, Option.of(nextChargeCoefficient),
                Option.empty());
    }

    private boolean validateExecutionIsNeeded(BalancePaymentInfo paymentInfo, Currency currency,
                                              ListF<GroupTrustPaymentRequest> recentPayments,
                                              Option<CardEntity> cardO) {
        String messagePrefix = String.format("Auto payment not possible for client %s and currency %s: ",
                paymentInfo.getClientId(), currency);

        return isOk(validateAutoBillingEnabled(paymentInfo), messagePrefix)
                && isOk(validateCard(paymentInfo, cardO), messagePrefix)
                && isOk(validatePayerIsAdminInOrgs(paymentInfo), messagePrefix)
                && isOk(validateBalanceForRegularPayment(paymentInfo, currency), messagePrefix)
                && isOk(validateNoPostpaidProduct(paymentInfo, currency), messagePrefix)
                && isOk(validateInitPaymentWithoutTransactionId(recentPayments), messagePrefix)
                && isOk(validateNoActivePayment(recentPayments), messagePrefix)
                && isOk(validateNoRecentSuccessPayment(recentPayments), messagePrefix)
                && isOk(validateRecentCancelledPayment(recentPayments, cardO.get()), messagePrefix)
                && isOk(validatePaymentsNotTooFrequent(recentPayments), messagePrefix);
    }

    private ActionResult validateBalanceForRegularPayment(BalancePaymentInfo paymentInfo, Currency currency) {
        Long clientId = paymentInfo.getClientId();

        ClientBalanceEntity balance = clientBalanceDao.find(clientId, currency)
                .orElseThrow(() -> new IllegalStateException(
                        String.format("no balance found for payment_info %s", paymentInfo)));
        if (balance.getBalanceVoidAt().isEmpty()
                || balance.getBalanceVoidAt().get().minus(settings.getAutoPayLeadTime()).isAfterNow()) {

            return ActionResult.fail(String.format("balance void date %s is too far from now to charge payment",
                    balance.getBalanceVoidAt()));
        }

        return ActionResult.success();
    }
}
