package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.UUID;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.base.AbstractInstant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks.GroupAutoReBillingTask;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.PriceOverrideReason;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentGroupServiceInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupServicesManager;
import ru.yandex.chemodan.app.psbilling.core.groups.SubscriptionContext;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class AutoResurrectionPayManager extends BasePayManager {
    private static final Logger logger = LoggerFactory.getLogger(AutoPayManager.class);
    private final GroupServicesManager groupServicesManager;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;

    public AutoResurrectionPayManager(ClientBalanceDao clientBalanceDao, Settings settings,
                                      GroupTrustPaymentRequestDao groupTrustPaymentRequestDao,
                                      GroupBillingService groupBillingService, GroupDao groupDao,
                                      ClientBalanceCalculator clientBalanceCalculator, CardDao cardDao,
                                      GroupServiceDao groupServiceDao, GroupProductManager groupProductManager,
                                      BalanceService balanceService, TaskScheduler taskScheduler,
                                      FeatureFlags featureFlags, GroupServicesManager groupServicesManager,
                                      DirectoryService directoryService,
                                      GroupServicePriceOverrideDao groupServicePriceOverrideDao) {
        super(clientBalanceDao, settings, groupTrustPaymentRequestDao, groupBillingService, groupDao,
                clientBalanceCalculator, cardDao, groupServiceDao, groupProductManager, balanceService,
                taskScheduler, featureFlags, directoryService);
        this.groupServicesManager = groupServicesManager;
        this.groupServicePriceOverrideDao = groupServicePriceOverrideDao;
    }


    protected Duration getAutoPayRecentCanceledPaymentsLookupTime() {
        return settings.getAutoResurrectMinRetryIntervalMinutes();
    }

    public void scheduleAutoResurrectionPayment() {
        if (!featureFlags.getAutoResurrectionPayAutoOn().isEnabled()) {
            logger.info("auto resurrection payment disabled");
            return;
        }

        Instant balanceVoidAfter = Instant.now().minus(settings.getAutoResurrectMaxResurrectTimeMinutes());
        BatchFetchingUtils.<Tuple2<Long, Currency>, Tuple2<Long, Currency>>fetchAndProcessBatchedEntities(
                (batchSize, lastData) ->
                        clientBalanceDao.findClientsForAutoResurrectionPay(balanceVoidAfter,
                                Instant.now().minus(getAutoPayRecentCanceledPaymentsLookupTime()),
                                settings.getAutoPayChargePlan().last(),
                                batchSize, lastData),
                Function.identityF(),
                clientAndCurrency ->
                        taskScheduler.schedule(new GroupAutoReBillingTask(clientAndCurrency._1, clientAndCurrency._2)),
                CLIENTS_FETCH_BATCH_SIZE,
                logger);
    }

    // charge payment for already disabled services
    public void chargeResurrectionAutoPayment(Long clientId, Currency currency) {
        BalancePaymentInfo paymentInfo = getPaymentInfo(clientId);
        if (!featureFlags.getAutoResurrectionPayAutoOn().isEnabledForUid(paymentInfo.getPassportUid())) {
            logger.info("auto resurrection payment disabled");
            return;
        }

        logger.info("charging resurrection payment for clientId {} and currency {}", clientId, currency);
        ListF<GroupTrustPaymentRequest> recentPayments = getRecentPayments(clientId);
        Option<CardEntity> cardO = cardDao.findB2BPrimary(paymentInfo.getPassportUid());

        if (!validateExecutionIsNeeded(paymentInfo, currency, recentPayments, cardO)) {
            return;
        }

        CardEntity card = cardO.orElseThrow();
        ListF<GroupService> servicesToResurrect = getServicesToResurrect(paymentInfo, currency);
        if (servicesToResurrect.size() == 0) {
            logger.error("no services to resurrect for clientId {} and currency {}", clientId, currency);
            throw new IllegalStateException("unable to collect services to resurrect");
        }

        GroupTrustPaymentGroupServiceInfo paymentServiceInfo =
                new GroupTrustPaymentGroupServiceInfo(servicesToResurrect.map(groupServicesManager::wrap));

        Double nextChargeCoefficient = getNextChargeCoefficient(recentPayments, card.getId());
        Money chargeAmount = getChargeAmount(servicesToResurrect, currency, nextChargeCoefficient);

        if (chargeAmount.getAmount().compareTo(BigDecimal.ZERO) == 0) {
            logger.error("no charge required cause total price of all services to resurrect is zero " +
                    "for client {} and currency {}; services: {}", clientId, currency, servicesToResurrect);
            throw new IllegalStateException("no paid services found to resurrect");
        }

        groupBillingService.chargePayment(paymentInfo.getPassportUid(), clientId, chargeAmount, card,
                Option.of(nextChargeCoefficient),
                Option.of(paymentServiceInfo));
    }

    public void resurrectServices(long clientId, Currency currency,
                                  GroupTrustPaymentGroupServiceInfo groupServiceInfo) {
        ListF<GroupService> allDisabledServices = groupServiceDao.findLastDisabledPrepaidServices(clientId, currency);
        MapF<UUID, ListF<GroupServicePriceOverride>> allPriceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(allDisabledServices);

        for (GroupTrustPaymentGroupServiceInfo.ActivateServiceInfo serviceInfo :
                groupServiceInfo.getActivateServices()) {
            Group group = groupDao.findById(serviceInfo.getGroupId());
            if (group.getPaymentInfo().isEmpty() || !group.getPaymentInfo().get().getClientId().equals(clientId)) {
                logger.warn("group {} payment info id has changed", group);
                continue;
            }

            ListF<GroupService> activeServices =
                    groupServicesManager.find(serviceInfo.getGroupId(), Target.ENABLED).filter(x -> !x.isHidden());
            if (activeServices.size() > 0) {
                logger.warn("group {} already have paid services {}", serviceInfo.getGroupId(), activeServices);
                continue;
            }

            boolean groupIsEdu = directoryService.getOrganizationFeatures(group).isEdu360Active();

            for (GroupTrustPaymentGroupServiceInfo.Product productCode : serviceInfo.getProducts()) {
                resurrectProduct(group, groupIsEdu, productCode, allDisabledServices, allPriceOverrides);
            }
        }
    }

    protected ActionResult validateNoActiveService(BalancePaymentInfo paymentInfo, Currency currency) {
        ListF<GroupService> activeServices =
                groupServiceDao.findActiveGroupServicesByPaymentInfoClient(paymentInfo.getClientId(), currency)
                        .filter(gs -> !gs.isHidden() && !gs.isSkipTransactionsExport());

        MapF<UUID, GroupProduct> groupProductsById =
                groupProductManager.findByIds(activeServices.map(GroupService::getGroupProductId))
                        .toMapMappingToKey(GroupProduct::getId);

        ListF<GroupService> paidActiveServices =
                activeServices.filter(gs -> !groupProductsById.getTs(gs.getGroupProductId()).isFree());

        if (paidActiveServices.size() > 0) {
            return ActionResult.fail(String.format("found active services %s", paidActiveServices));
        }

        return ActionResult.success();
    }

    private void resurrectProduct(Group group, boolean groupIsEdu,
                                  GroupTrustPaymentGroupServiceInfo.Product productCode,
                                  ListF<GroupService> allDisabledServices,
                                  MapF<UUID, ListF<GroupServicePriceOverride>> allOverrides) {
        logger.info("creating service for group {} and product {}", group, productCode);
        GroupProduct product = groupProductManager.findProduct(productCode.getCode());
        GroupService service = groupServicesManager.createGroupService(
                new SubscriptionContext(product, group, group.getPaymentInfo().get().getPassportUid(),
                        Option.empty(), groupIsEdu));

        Option<GroupService> disabledServiceO = allDisabledServices.filter(
                x -> x.getGroupId().equals(group.getId()) && x.getGroupProductId().equals(product.getId())).firstO();
        ListF<GroupServicePriceOverride> actualOverrides = getActualOverrides(allOverrides,
                disabledServiceO);
        logger.info("found actual overrides: {}", actualOverrides);

        for (GroupServicePriceOverride override : actualOverrides) {
            GroupServicePriceOverride priceOverride =
                    groupServicePriceOverrideDao.insert(GroupServicePriceOverrideDao.InsertData.builder()
                            .groupServiceId(service.getId())
                            .startDate(override.getStartDate().isBeforeNow()
                                    ? service.getCreatedAt() : override.getStartDate())
                            .endDate(override.getEndDate())
                            .hidden(override.isHidden())
                            .pricePerUserInMonth(override.getPricePerUserInMonth())
                            .reason(override.getReason())
                            .trialUsageId(override.getTrialUsageId())
                            .build());
            logger.info("created price override priceOverride {}", priceOverride);
        }

        logger.info("service has been resurrected: {}", service);
    }

    private ListF<GroupServicePriceOverride> getActualOverrides(MapF<UUID,
            ListF<GroupServicePriceOverride>> allOverrides, Option<GroupService> disabledServiceO) {
        if (disabledServiceO.isPresent()) {
            return allOverrides.getOrElse(disabledServiceO.get().getId(), Cf.list())
                    // trials are already covered by trial mechanism on service activation
                    .filter(x -> !x.getReason().equals(PriceOverrideReason.TRIAL)
                            && x.getEndDate().map(AbstractInstant::isAfterNow).orElse(true));
        }
        return Cf.list();
    }

    private boolean validateExecutionIsNeeded(BalancePaymentInfo paymentInfo, Currency currency,
                                              ListF<GroupTrustPaymentRequest> recentPayments,
                                              Option<CardEntity> cardO) {
        String messagePrefix = String.format("Auto payment not possible for client %s and currency %s: ",
                paymentInfo.getClientId(), currency);

        return isOk(validateAutoBillingEnabled(paymentInfo), messagePrefix)
                && isOk(validateCard(paymentInfo, cardO), messagePrefix)
                && isOk(validatePayerIsAdminInOrgs(paymentInfo), messagePrefix)
                && isOk(validateBalance(paymentInfo, currency), messagePrefix)
                && isOk(validateNoActiveService(paymentInfo, currency), messagePrefix)
                && isOk(validateInitPaymentWithoutTransactionId(recentPayments), messagePrefix)
                && isOk(validateNoActivePayment(recentPayments), messagePrefix)
                && isOk(validateNoRecentSuccessPayment(recentPayments), messagePrefix)
                && isOk(validateRecentCancelledPayment(recentPayments, cardO.get()), messagePrefix)
                && isOk(validatePaymentsNotTooFrequent(recentPayments), messagePrefix);
    }

    private ActionResult validateBalance(BalancePaymentInfo paymentInfo, Currency currency) {
        Long clientId = paymentInfo.getClientId();

        ClientBalanceEntity balance = clientBalanceDao.find(clientId, currency)
                .orElseThrow(() -> new IllegalStateException(
                        String.format("no balance found for payment_info %s", paymentInfo)));
        if (balance.getBalanceVoidAt().isEmpty()) {
            return ActionResult.fail(String.format("no void date for balance %s", balance));
        }

        if (balance.getBalanceAmount().compareTo(BigDecimal.ZERO) > 0) {
            return ActionResult.fail(String.format("balance is already positive %s", balance));
        }

        return ActionResult.success();
    }

    private ListF<GroupService> getServicesToResurrect(BalancePaymentInfo paymentInfo, Currency currency) {
        return groupServiceDao.findLastDisabledPrepaidServices(paymentInfo.getClientId(), currency);
    }

    @Override
    protected Duration getMinRetryInterval() {
        return settings.getAutoResurrectMinRetryIntervalMinutes();
    }
}
