package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.UUID;

import com.google.common.collect.Ordering;
import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentInitiationType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.balanceclient.model.response.CheckRequestPaymentResponse;
import ru.yandex.chemodan.balanceclient.model.response.GetBoundPaymentMethodsResponse;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public abstract class BasePayManager {
    private static final Logger logger = LoggerFactory.getLogger(BasePayManager.class);
    protected static final Integer CLIENTS_FETCH_BATCH_SIZE = 2000;

    protected ClientBalanceDao clientBalanceDao;
    protected Settings settings;
    protected GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;
    protected GroupBillingService groupBillingService;
    protected GroupDao groupDao;
    protected ClientBalanceCalculator clientBalanceCalculator;
    protected CardDao cardDao;
    protected GroupServiceDao groupServiceDao;
    protected GroupProductManager groupProductManager;
    protected BalanceService balanceService;
    protected TaskScheduler taskScheduler;
    protected FeatureFlags featureFlags;
    protected DirectoryService directoryService;

    protected boolean isOk(ActionResult validationResult, String messagePrefix) {
        if (!validationResult.isSuccess()) {
            logger.info(messagePrefix + validationResult.getMessage());
            return false;
        }
        return true;
    }

    protected ActionResult validateAutoBillingEnabled(BalancePaymentInfo paymentInfo) {
        if (!paymentInfo.isB2bAutoBillingEnabled()) {
            return ActionResult.fail("auto payment is disabled");
        }
        return ActionResult.success();
    }

    protected ActionResult validateNoPostpaidProduct(BalancePaymentInfo paymentInfo, Currency currency) {
        Option<GroupProduct> postPaidProduct =
                groupProductManager.findByIds(groupServiceDao.findActiveGroupServicesByPaymentInfoClient(
                                paymentInfo.getClientId(), currency)
                        .map(GroupService::getGroupProductId).unique()).filter(x -> !x.isFree() && !x.isPrepaid()).firstO();
        if (postPaidProduct.isPresent()) {
            return ActionResult.fail(String.format("found active postpaid product %s", postPaidProduct));
        }

        return ActionResult.success();
    }

    protected ActionResult validateNoActivePayment(ListF<GroupTrustPaymentRequest> recentPayments) {
        Option<GroupTrustPaymentRequest> activePayment = findLastPayment(recentPayments, Option.empty(),
                PaymentRequestStatus.INIT, settings.getAutoPayRecentInitPaymentsLookupTime());
        if (activePayment.isPresent()) {
            return ActionResult.fail(String.format("found active payment %s", activePayment.get()));
        }

        return ActionResult.success();
    }
    protected ActionResult validateInitPaymentWithoutTransactionId(ListF<GroupTrustPaymentRequest> recentPayments) {
        if(recentPayments.exists(p -> p.getStatus() == PaymentRequestStatus.INIT && p.getTransactionId().isEmpty())){
            return ActionResult.fail(String.format("found init payment without transaction id %s", recentPayments));
        }

        return ActionResult.success();
    }

    protected ActionResult validateNoRecentSuccessPayment(ListF<GroupTrustPaymentRequest> recentPayments) {

        Option<GroupTrustPaymentRequest> successPayment = findLastPayment(recentPayments,
                Option.of(PaymentInitiationType.AUTO),
                PaymentRequestStatus.SUCCESS, settings.getAutoPayRecentSuccessPaymentsLookupTime());
        if (successPayment.isPresent()) {
            return ActionResult.fail(String.format("found recent success auto payment %s",
                    successPayment.get()));
        }

        return ActionResult.success();
    }

    protected ActionResult validateRecentCancelledPayment(ListF<GroupTrustPaymentRequest> recentPayments,
                                                          CardEntity card) {

        Option<GroupTrustPaymentRequest> canceledPayment = getLastCanceledPayment(recentPayments, card.getId());
        if (canceledPayment.isPresent()) {
            String paymentError = canceledPayment.get().getError().orElse("null error for canceled payment");
            if (!isNoFundsError(paymentError)) {
                return ActionResult.fail(
                        String.format("recent payment %s was not successful for card %s. " +
                                        "No further tries will happen. Error %s",
                                canceledPayment.get(), card, paymentError));
            }

            if (canceledPayment.get().getPaymentPeriodCoefficient().get().equals(settings.getAutoPayChargePlan().last())) {
                return ActionResult.fail(
                        String.format("recent payment %s was last in retry sequence. Will try later",
                                canceledPayment.get()));
            }
        }

        return ActionResult.success();
    }

    protected abstract Duration getMinRetryInterval();

    protected ActionResult validatePaymentsNotTooFrequent(ListF<GroupTrustPaymentRequest> recentPayments) {
        Double lastPaymentPlanCoefficient = settings.getAutoPayChargePlan().last();
        Duration lookupDuration = getMinRetryInterval();
        ListF<GroupTrustPaymentRequest> canceledPayments = findLastPayments(recentPayments,
                Option.of(PaymentInitiationType.AUTO),
                PaymentRequestStatus.CANCELLED, lookupDuration);

        Option<GroupTrustPaymentRequest> canceledPayment =
                // Ищем последнюю попытку списать деньги.
                // Это или последняя попытка снять минимум денег, или последняя неизвестная ошибка при снятии
                canceledPayments.find(p ->
                        (p.getError().isPresent()
                                && isNoFundsError(p.getError().get())
                                && p.getPaymentPeriodCoefficient().get().equals(lastPaymentPlanCoefficient)
                        )
                                || p.getError().isEmpty()
                                || !isNoFundsError(p.getError().get())
                );
        if (canceledPayment.isPresent()) {
            return ActionResult.fail(
                    String.format("found recent auto payment try %s within period %s",
                            canceledPayment.get(), lookupDuration));
        }
        return ActionResult.success();
    }

    protected ActionResult validateCard(BalancePaymentInfo paymentInfo, Option<CardEntity> cardO) {
        PassportUid uid = paymentInfo.getPassportUid();

        if (cardO.isEmpty()) {
            return ActionResult.fail(String.format("no b2b primary card found for uid %s", uid));
        }

        CardEntity card = cardO.get();
        if (!card.getStatus().equals(CardStatus.ACTIVE)) {
            return ActionResult.fail(String.format("b2b primary card %s for uid %s is not active", card, uid));
        }

        ListF<GetBoundPaymentMethodsResponse> cardsFromBalance =
                Cf.arrayList(balanceService.getBoundPaymentMethods(uid.getUid()));
        Option<GetBoundPaymentMethodsResponse> cardInBalanceO =
                cardsFromBalance.find(x -> x.getPaymentMethodId().equals(card.getExternalId()));
        if (cardInBalanceO.isEmpty()) {
            cardDao.updateStatus(card.getId(), CardStatus.DISABLED);
            groupBillingService.setAutoBillingOff(paymentInfo);
            return ActionResult.fail(String.format("card %s is no longer available for uid %s", card, uid));
        } else if (cardInBalanceO.get().isExpired()) {
            cardDao.updateStatus(card.getId(), CardStatus.EXPIRED);
            groupBillingService.setAutoBillingOff(paymentInfo);
            return ActionResult.fail(String.format("card %s is expired", card));
        }

        return ActionResult.success();
    }

    protected ActionResult validatePayerIsAdminInOrgs(BalancePaymentInfo paymentInfo) {
        if (!featureFlags.getAutoPayDisableOnUserIsNotAdmin().isEnabled()) {
            return ActionResult.success();
        }

        PassportUid uid = paymentInfo.getPassportUid();
        if (!directoryService.isPayerAdminInOrgs(uid)) {
            groupBillingService.setAutoBillingOff(paymentInfo);
            return ActionResult.fail(String.format("payer %s isn't admin in groups", uid.toString()));
        }

        return ActionResult.success();
    }

    protected ListF<GroupTrustPaymentRequest> getRecentPayments(Long clientId) {
        Duration lookupTime = Ordering.natural().max(
                settings.getAutoPayRecentInitPaymentsLookupTime(),
                settings.getAutoPayRecentCanceledPaymentsLookupTime(),
                settings.getAutoPayRecentSuccessPaymentsLookupTime());
        return groupTrustPaymentRequestDao.findRecentPayments(clientId, Option.of(Instant.now().minus(lookupTime)));
    }

    protected BalancePaymentInfo getPaymentInfo(Long clientId) {
        SetF<Option<BalancePaymentInfo>> uniquePaymentInfo =
                groupDao.findGroupsByPaymentInfoClient(clientId).map(Group::getPaymentInfo).unique();
        if (uniquePaymentInfo.size() > 1) {
            throw new IllegalStateException(String.format("found not unique payment info for clientId %s",
                    clientId));
        }
        if (uniquePaymentInfo.isEmpty()) {
            throw new IllegalStateException(String.format("no payment info found for client %s", clientId));
        }
        return uniquePaymentInfo.iterator().next().orElseThrow(
                () -> new IllegalStateException(String.format("no payment info found for client %s", clientId)));
    }

    protected Option<GroupTrustPaymentRequest> findLastPayment(ListF<GroupTrustPaymentRequest> recentPayments,
                                                               Option<PaymentInitiationType> typeO,
                                                               PaymentRequestStatus status, Duration lookupDuration) {
        return findLastPayments(recentPayments, typeO, status, lookupDuration).firstO();
    }

    protected ListF<GroupTrustPaymentRequest> findLastPayments(ListF<GroupTrustPaymentRequest> recentPayments,
                                                               Option<PaymentInitiationType> typeO,
                                                               PaymentRequestStatus status, Duration lookupDuration) {
        return recentPayments.filter(x -> x.getStatus().equals(status)
                        && (typeO.isEmpty() || x.getPaymentInitiationType().equals(typeO.get()))
                        && Instant.now().minus(lookupDuration).isBefore(x.getCreatedAt()))
                .sortedByDesc(AbstractEntity::getCreatedAt);
    }


    protected Option<GroupTrustPaymentRequest> getLastCanceledPayment(ListF<GroupTrustPaymentRequest> recentPayments,
                                                                      UUID cardId) {
        Option<GroupTrustPaymentRequest> canceledPayment = findLastPayment(recentPayments,
                Option.of(PaymentInitiationType.AUTO),
                PaymentRequestStatus.CANCELLED, settings.getAutoPayRecentCanceledPaymentsLookupTime());
        if (canceledPayment.isPresent()) {
            logger.info("found recent canceled auto payment {}", canceledPayment.get());
            UUID paymentCardId = canceledPayment.get().getCardId().orElseThrow(
                    () -> new IllegalStateException("null card is unexpected for automatic payment"));
            if (paymentCardId.equals(cardId)) {
                return canceledPayment;
            }
        }
        return Option.empty();
    }

    @NotNull
    protected Money getChargeAmount(ListF<GroupService> groupServices, Currency currency,
                                    Double nextChargeCoefficient) {
        Option<ClientBalanceCalculator.PricePeriod> pricePeriod =
                clientBalanceCalculator.calculatePricePeriods(Instant.now(), groupServices, currency)
                        .filter(x -> x.to.isAfter(Instant.now().plus(settings.getAutoPayLeadTime()))).firstO();
        BigDecimal pricePerMonth = pricePeriod.map(x -> x.price).orElse(BigDecimal.ZERO);
        return new Money(BigDecimal.valueOf(nextChargeCoefficient).multiply(pricePerMonth), currency);
    }

    protected Double getNextChargeCoefficient(ListF<GroupTrustPaymentRequest> recentPayments, UUID cardId) {
        ListF<Double> chargePlan = settings.getAutoPayChargePlan();
        Double nextChargeCoeff = chargePlan.first();
        Option<GroupTrustPaymentRequest> canceledPayment = getLastCanceledPayment(recentPayments, cardId);
        if (canceledPayment.isPresent()) {
            String paymentError = canceledPayment.get().getError().orElse("null error for canceled payment");
            if (isNoFundsError(paymentError)) {
                Option<Double> prevChargePeriod = canceledPayment.get().getPaymentPeriodCoefficient();
                if (prevChargePeriod.isPresent() && chargePlan.indexOfTs(prevChargePeriod.get()) < chargePlan.length() - 1) {
                    nextChargeCoeff = chargePlan.get(chargePlan.indexOfTs(prevChargePeriod.get()) + 1);
                }
            }
        }
        return nextChargeCoeff;
    }

    protected boolean isNoFundsError(String error) {
        return CheckRequestPaymentResponse.NO_MONEY_ERRORS.containsTs(error);
    }
}
