package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Comparator;
import java.util.Currency;
import java.util.UUID;

import com.google.common.collect.Ordering;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.ToString;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.joda.time.ReadableInstant;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.CorpContract;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks.UpdateClientBalanceTask;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.directory.DirectoryService;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransaction;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.TooBigGroupException;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.chemodan.app.psbilling.core.util.MathUtils;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ClientBalanceCalculator {
    private static final Integer CLIENTS_FETCH_BATCH_SIZE = 2000;
    private static final Logger logger = LoggerFactory.getLogger(ClientBalanceCalculator.class);

    private final ClientBalanceDao clientBalanceDao;
    private final GroupServiceDao groupServiceDao;
    private final GroupServiceMemberDao groupServiceMemberDao;
    private final GroupProductManager groupProductManager;
    private final GroupBalanceService groupBalanceService;
    private final GroupServiceTransactionsDao groupServiceTransactionsDao;
    private final DirectoryService directoryService;
    private final GroupDao groupDao;
    private final TaskScheduler taskScheduler;
    private final Settings settings;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;

    public void scheduleUpdateObsoleteClientBalances() {
        Instant staleBalanceDate = Instant.now().minus(settings.getBalanceStaleIntervalHours());
        Instant staleInvoiceDate = Instant.now().minus(settings.getBalanceInvoiceStaleIntervalHours());

        BatchFetchingUtils.<Long, Long>fetchAndProcessBatchedEntities(
                (batchSize, lastClientId) ->
                        clientBalanceDao.findObsoleteBalanceClients(staleBalanceDate, staleInvoiceDate, lastClientId,
                                batchSize),
                Function.identityF(),
                clientId -> taskScheduler.schedule(new UpdateClientBalanceTask(clientId)),
                CLIENTS_FETCH_BATCH_SIZE,
                logger);
    }

    @Transactional
    public MapF<Currency, ClientBalanceEntity> updateClientBalance(Long clientId) {
        logger.info("updating balance for client {}", clientId);
        MapF<Currency, ClientBalanceInfo> clientBalances = calculateClientBalanceWithTransactions(clientId);
        return clientBalances.mapValues(cb -> updateClientBalance(clientId, cb));
    }

    @Transactional
    public ClientBalanceEntity updateClientBalance(Long clientId, Currency currency) {
        MapF<Currency, ClientBalanceEntity> updatedBalances = updateClientBalance(clientId);
        if (!updatedBalances.containsKeyTs(currency)) {
            ClientBalanceInfo clientBalanceInfo = new ClientBalanceInfo(clientId, currency);
            return updateClientBalance(clientId, clientBalanceInfo);
        }
        return updatedBalances.getTs(currency);
    }

    @Transactional
    public MapF<Currency, ClientBalanceEntity> updateClientBalance(Long clientId,
                                                                   ListF<CorpContract> contractsBalance) {
        logger.info("updating balance for client {} with balance Items", clientId, contractsBalance);
        MapF<Currency, ClientBalanceInfo> clientBalances = calculateClientBalanceWithTransactions(clientId,
                contractsBalance);
        return clientBalances.mapValues(cb -> updateClientBalance(clientId, cb));
    }

    private ClientBalanceEntity updateClientBalance(Long clientId, ClientBalanceInfo clientBalanceInfo) {
        Option<ClientBalanceEntity> oldBalance = clientBalanceDao.find(clientId,
                clientBalanceInfo.getCurrency());
        Option<Instant> voidDate = calculateVoidDate(clientBalanceInfo,
                oldBalance.flatMapO(ClientBalanceEntity::getBalanceVoidAt));
        ClientBalanceEntity newBalance = clientBalanceDao.createOrUpdate(clientId, clientBalanceInfo.getCurrency(),
                clientBalanceInfo.getBalance(), voidDate);
        groupServiceDao.updatePrepaidNextBillingDateForClient(clientId, clientBalanceInfo.getCurrency(), voidDate);
        return newBalance;
    }

    public void updateVoidDate(GroupService groupService) {
        GroupProduct groupProduct = groupProductManager.findById(groupService.getGroupProductId());
        if (!groupProduct.isPrepaid()) {
            return;
        }
        Group group = groupDao.findById(groupService.getGroupId());
        if (group.getPaymentInfo().isEmpty()) {
            logger.info("Group %s has no payment info", group);
            return;
        }

        Long clientId = group.getPaymentInfo().get().getClientId();
        Currency currency = groupProduct.getPriceCurrency();
        Option<ClientBalanceEntity> clientBalanceO = clientBalanceDao.find(clientId, currency);
        if (clientBalanceO.isEmpty()) {
            logger.info("Group %s has no client balance", group);
            return;
        }
        ClientBalanceEntity clientBalance = clientBalanceO.get();

        Option<Instant> voidDate = calculateVoidDate(
                new ClientBalanceInfo(clientId, currency, clientBalance.getBalanceAmount()),
                clientBalance.getBalanceVoidAt());
        clientBalanceDao.updateVoidAt(clientBalance.getId(), voidDate);
        groupServiceDao.updatePrepaidNextBillingDateForClient(clientId, groupProduct.getPriceCurrency(), voidDate);
    }

    public Option<Instant> calculateVoidDate(ClientBalanceInfo clientBalanceInfo, Option<Instant> oldVoidDateO) {
        Long clientId = clientBalanceInfo.getClientId();
        logger.info("calculate void date for client {}; current balance is {}; oldVoidDate is {}",
                clientId, clientBalanceInfo, oldVoidDateO);
        ListF<GroupService> groupServices = groupServiceDao.findActiveGroupServicesByPaymentInfoClient(
                clientId, clientBalanceInfo.getCurrency());

        Instant balanceActualDate = clientBalanceInfo.getLastTransactionsDate().orElse(Instant.now());
        Instant oldVoidDate = oldVoidDateO.orElse(balanceActualDate);
        BigDecimal balance = clientBalanceInfo.getBalance();
        if (balance.compareTo(BigDecimal.ZERO) < 0) {
            logger.info("client {} balance is below zero: {}", clientBalanceInfo.getClientId(), balance);
            return Option.of(oldVoidDate);
        }

        if (groupServices.length() == 0) {
            if (balance.compareTo(BigDecimal.ZERO) > 0) {
                logger.info("no active services for client {} and amount is greater than 0: {}", clientId, balance);
                return Option.empty();
            }
            return Option.of(oldVoidDate);
        }

        Option<Instant> result = Option.empty();
        ListF<PricePeriod> pricePeriods = calculatePricePeriods(balanceActualDate, groupServices,
                clientBalanceInfo.getCurrency());
        for (PricePeriod pricePeriod : pricePeriods) {
            Tuple2<BigDecimal, Instant> balanceAndDate = calculateBalanceDateWithinPeriod(clientId, balance,
                    pricePeriod);
            balance = balanceAndDate._1;
            if (MathUtils.areNear(balance, BigDecimal.ZERO, 2)) {
                Instant balanceDate = DateTimeUtils.MAX_INSTANT.equals(balanceAndDate._2) ? oldVoidDate : balanceAndDate._2;

                result = result.filter(inst -> inst.isAfter(balanceDate))
                        .orElse(Option.of(balanceDate));

                if (pricePeriod.price.compareTo(BigDecimal.ZERO) > 0) {
                    break;
                }
            }
        }

        logger.info("Void date {} calculated for balance {}", result, clientBalanceInfo);
        return result;
    }

    private Tuple2<BigDecimal, Instant> calculateBalanceDateWithinPeriod(Long clientId, BigDecimal balance,
                                                                         PricePeriod pricePeriod) {
        if (MathUtils.areNear(pricePeriod.price, BigDecimal.ZERO, 2)) {
            logger.info("group service price is zero for client {} and period {}", clientId, pricePeriod);
            return Tuple2.tuple(balance, pricePeriod.to);
        }

        Tuple2<BigDecimal, Instant> balanceAndDate =
                handleCurMonthEnding(clientId, pricePeriod.from, pricePeriod.to, pricePeriod.price, balance);
        // end at curren month?
        if (MathUtils.areNear(balanceAndDate._1, BigDecimal.ZERO, 2) || balanceAndDate._2.equals(pricePeriod.to)) {
            return balanceAndDate;
        }

        // has money to pay next several months?
        balanceAndDate = handleFullMonthEnding(clientId, balanceAndDate._2, pricePeriod.to, pricePeriod.price,
                balanceAndDate._1);
        if (MathUtils.areNear(balanceAndDate._1, BigDecimal.ZERO, 2) || balanceAndDate._2.equals(pricePeriod.to)) {
            return balanceAndDate;
        }

        // handle last month
        balanceAndDate =
                handleCurMonthEnding(clientId, balanceAndDate._2, pricePeriod.to, pricePeriod.price, balanceAndDate._1);

        logger.info("client's {} balance is {} and void date is {} after period {}", clientId, balanceAndDate._1,
                balanceAndDate._2, pricePeriod);
        return balanceAndDate;
    }

    private Tuple2<BigDecimal, Instant> handleFullMonthEnding(Long clientId, Instant dateFrom, Instant maxDate,
                                                              BigDecimal pricePerMonth, BigDecimal balance) {
        logger.info("calc void date for full paid months {} for client {} with max date limit {}; " +
                        "cur balance is {}, price per month is {}",
                dateFrom, clientId, maxDate, balance, pricePerMonth);
        Validate.isTrue(dateFrom.equals(DateTimeUtils.getMonthFirstDate(dateFrom).toInstant()));

        // for full paid month we don't have to calculate price per second
        BigDecimal paidMonthsLeft = balance.divide(pricePerMonth, 1, RoundingMode.HALF_UP);
        logger.info("left months count: {}", paidMonthsLeft);
        int paidMonthsLeftRounded = paidMonthsLeft.setScale(0, RoundingMode.DOWN).intValue();

        // balance on last month 1st day
        DateTime voidDate = dateFrom.toDateTime().plusMonths(paidMonthsLeftRounded);
        if (voidDate.isAfter(maxDate)) {
            paidMonthsLeftRounded = new Period(dateFrom, maxDate).getMonths();
            logger.info("limited left months count to: {}", paidMonthsLeftRounded);
            voidDate = DateTimeUtils.getMonthFirstDate(maxDate);
        }
        balance = balance.subtract(pricePerMonth.multiply(BigDecimal.valueOf(paidMonthsLeftRounded)));
        logger.info("balance on last month {} is {}", voidDate, balance);
        return Tuple2.tuple(balance, voidDate.toInstant());
    }

    private Tuple2<BigDecimal, Instant> handleCurMonthEnding(Long clientId, Instant dateFrom, Instant maxDate,
                                                             BigDecimal pricePerMonth, BigDecimal balance) {
        logger.info("calc void date for month {} for client {} with max date limit {}; " +
                        "cur balance is {}, price per month is {}",
                dateFrom, clientId, maxDate, balance, pricePerMonth);
        BigDecimal pricePerMilliSecond = getMonthPricePerMillisecond(dateFrom.toDateTime(), pricePerMonth);

        Instant nextMonthFirstDate = DateTimeUtils.getNextMonthFirstDate(dateFrom).toInstant();
        if (maxDate.isAfter(nextMonthFirstDate)) {
            maxDate = nextMonthFirstDate;
            logger.info("date limit is after cur month's end. new date limit is {}", maxDate);
        }
        BigDecimal curMonthLeftMilliSeconds = getLeftMilliseconds(dateFrom, maxDate);
        BigDecimal curMonthPrice = curMonthLeftMilliSeconds.multiply(pricePerMilliSecond);

        logger.info("pricePerMilliSecond = {}; curMonthLeftMilliSeconds={}; curMonthPrice={}",
                pricePerMilliSecond, curMonthLeftMilliSeconds, curMonthPrice);

        // not enough money to pay all month period
        if (curMonthPrice.compareTo(balance) >= 0) {
            logger.info("current month price {} is bigger then balance {}", curMonthPrice, balance);
            BigDecimal millisecondsLeft = balance.divide(pricePerMilliSecond, 0, RoundingMode.HALF_UP);
            logger.info("milliseconds left in cur month: {}", millisecondsLeft);
            maxDate = dateFrom.plus(millisecondsLeft.longValue());
            curMonthPrice = millisecondsLeft.multiply(pricePerMilliSecond);
        }

        logger.info("client {} void date is {} (current month)", clientId, maxDate);
        return Tuple2.tuple(balance.subtract(curMonthPrice), maxDate);

    }

    public MapF<Currency, ClientBalanceInfo> calculateClientBalanceWithTransactions(Long clientId) {
        logger.info("calculate balance for client {}", clientId);
        ListF<CorpContract> balanceItems = groupBalanceService.getContractBalanceItems(clientId, true);
        return calculateClientBalanceWithTransactions(clientId, balanceItems);
    }

    public MapF<Currency, ClientBalanceInfo> calculateClientBalanceWithTransactions(Long clientId,
                                                                                    ListF<CorpContract> contractsBalance) {
        logger.info("calculate balance for client {} and balances {}", clientId, contractsBalance);
        MapF<Currency, ClientBalanceInfo> balanceAmounts =
                groupBalanceService.getBalanceAmount(clientId, contractsBalance);
        return balanceAmounts.mapValues(this::addTransactionsInfo);
    }

    public static ListF<PricePeriod> slitByPricesInterval(Interval interval,
                                                          ListF<GroupServicePriceOverride> priceOverrides,
                                                          BigDecimal defaultPrice) {
        ListF<PricePeriod> intervals = Cf.arrayList();

        boolean allIntervalProcessed = false;
        // price overrides sorted in natural order
        for (GroupServicePriceOverride priceOverride : priceOverrides) {
            if (interval.getEnd().isAfter(priceOverride.getStartDate())
                    && (!priceOverride.getEndDate().isPresent() ||
                    interval.getStart().isBefore(priceOverride.getEndDate().get()))) {
                // есть пересечение с priceOverride. вся часть интервала, которая до перечсечения - это подынтервал
                // с ценой по умолчанию. само пересечение - с ценой из priceOverride, все что после - это то,
                // что еще предстоит проанализировать на перекрытия
                if (interval.getStart().isBefore(priceOverride.getStartDate())) {
                    intervals.add(new PricePeriod(interval.getStart().toInstant(), priceOverride.getStartDate(),
                            defaultPrice));
                }

                ReadableInstant overlappingIntervalStart =
                        Ordering.natural().max(interval.getStart(), priceOverride.getStartDate());
                ReadableInstant overlappingIntervalEnd = !priceOverride.getEndDate().isPresent() ?
                        interval.getEnd() : Ordering.natural().min(interval.getEnd(), priceOverride.getEndDate().get());
                intervals.add(new PricePeriod(overlappingIntervalStart.toInstant(), overlappingIntervalEnd.toInstant(),
                        priceOverride.getPricePerUserInMonth()));

                if (overlappingIntervalEnd.isBefore(interval.getEnd())) {
                    interval = new Interval(overlappingIntervalEnd, interval.getEnd());
                } else {
                    allIntervalProcessed = true;
                }
            }
        }

        if (!allIntervalProcessed) {
            intervals.add(new PricePeriod(interval.getStart().toInstant(), interval.getEnd().toInstant(),
                    defaultPrice));
        }

        return intervals;
    }

    private ClientBalanceInfo addTransactionsInfo(ClientBalanceInfo balanceAmount) {
        logger.info("calculate client balance by info {}", balanceAmount);

        LocalDate transactionsNotBilledFromDate = balanceAmount.getLastActDate()
                .map(i -> i.toDateTime().plusDays(1))
                .orElse(new DateTime(0))
                .toLocalDate();

        logger.info("transactions are not billed from {}", transactionsNotBilledFromDate);

        ListF<GroupServiceTransaction> clientTransactions =
                groupServiceTransactionsDao.findClientTransactionsFrom(
                        transactionsNotBilledFromDate, balanceAmount.getClientId());
        BigDecimal transactionsSum = MathUtils.sum(clientTransactions.map(GroupServiceTransaction::getAmount));
        logger.info("transactions sum is {}", transactionsSum);
        balanceAmount.setTransactionsSum(transactionsSum);

        Option<Instant> transactionsLastCalcDate = Option.empty();
        Option<LocalDate> transactionsLastBillingDate = Option.empty();
        if (clientTransactions.length() > 0) {
            GroupServiceTransaction lastTransaction =
                    clientTransactions.max(Comparator.comparing(GroupServiceTransaction::getBillingDate));
            transactionsLastCalcDate = Option.of(lastTransaction.getCalculatedAt());
            transactionsLastBillingDate = Option.of(lastTransaction.getBillingDate());
        }

        logger.info("transactions last calc date is {}", transactionsLastCalcDate);
        logger.info("transactions last billing date is {}", transactionsLastBillingDate);

        Option<Instant> transactionsLastDate;
        // if transactions calculated for today - transactionsLastDate is transactionsLastCalcDate
        if (transactionsLastBillingDate.map(x -> x.equals(LocalDate.now())).orElse(false)) {
            transactionsLastDate = transactionsLastCalcDate;
        } else {
            transactionsLastDate = transactionsLastBillingDate.map(x -> DateTimeUtils.toInstant(x.plusDays(1)));
        }
        balanceAmount.setLastTransactionsDate(transactionsLastDate);
        logger.info("client's {} balance is {}", balanceAmount.getClientId(), balanceAmount);
        return balanceAmount;
    }

    public ListF<PricePeriod> calculatePricePeriods(Instant from, ListF<GroupService> groupServices,
                                                    Currency currency) {
        logger.info("calculate price periods for currency {} from {} for services {}", currency, from, groupServices);
        Interval interval = new Interval(from, DateTimeUtils.MAX_INSTANT);
        ListF<PricePeriod> result = Cf.arrayList();
        MapF<GroupService, ListF<PricePeriod>> prices = Cf.hashMap();

        MapF<UUID, GroupProduct> groupProducts =
                groupProductManager.findByIds(groupServices.map(GroupService::getGroupProductId))
                        .toMap(GroupProduct::getId, x -> x);

        groupServices =
                groupServices.filter(x -> groupProducts.getTs(x.getGroupProductId()).getPriceCurrency().equals(currency));
        MapF<UUID, ListF<GroupServicePriceOverride>> groupServicePriceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(groupServices.map(AbstractEntity::getId));
        logger.info("found group service overrides: {}", groupServicePriceOverrides);
        for (GroupService gs : groupServices) {
            Interval curInterval = gs.getCreatedAt().isAfter(interval.getStart())
                    ? interval.withStart(gs.getCreatedAt())
                    : interval;
            ListF<PricePeriod> gsPrices = slitByPricesInterval(curInterval,
                    groupServicePriceOverrides.getOrElse(gs.getId(), Cf.list()),
                    groupProducts.getTs(gs.getGroupProductId()).getPricePerUserInMonth());
            logger.info("for groupService {} found price periods: {}", gs, gsPrices);
            prices.put(gs, gsPrices);
        }

        ListF<Instant> startDates = prices.values().flatMap(pp -> pp).map(x -> x.from);
        ListF<Instant> endDates = prices.values().flatMap(pp -> pp).map(x -> x.to);
        ListF<Instant> milestones = startDates.plus(endDates).unique().sorted();

        for (int i = 1; i < milestones.length(); i++) {
            Instant prevMilestone = milestones.get(i - 1);
            Instant milestone = milestones.get(i);
            PricePeriod curPeriodPrice = new PricePeriod(prevMilestone, milestone, BigDecimal.ZERO);
            for (GroupService groupService : prices.keys()) {
                ListF<PricePeriod> gsPeriodPrices =
                        prices.getTs(groupService).filter(x -> !x.from.isAfter(prevMilestone) && !x.to.isBefore(milestone));
                if (gsPeriodPrices.size() == 0) {
                    logger.info("service {} is not applicable to period {}", groupService, curPeriodPrice);
                    continue;
                }
                PricePeriod gsPeriodPrice = gsPeriodPrices.single();
                BigDecimal gsPrice = gsPeriodPrice.price.multiply(BigDecimal.valueOf(getUserCount(groupService)));
                logger.info("price of groupService {} within period {}-{} is {}",
                        groupService, curPeriodPrice.from, curPeriodPrice.to, gsPrice);
                curPeriodPrice.price = curPeriodPrice.price.add(gsPrice);
            }
            result.add(curPeriodPrice);
        }
        logger.info("got price intervals: {}", result);
        return result;
    }

    private int getUserCount(GroupService gs) {
        MapF<UUID, Integer> groupMembersCount = Cf.hashMap();
        if (!gs.getStatus().equals(SynchronizationStatus.INIT) && gs.getTarget().equals(Target.ENABLED)) {
            return groupServiceMemberDao.getEnabledMembersCount(gs.getId());
        }
        // is status is in init state - no group service members can be found.
        // So we only can count on users count in Directory
        UUID groupId = gs.getGroupId();
        Option<Integer> memberCountO = groupMembersCount.getO(groupId);
        if (memberCountO.isEmpty()) {
            Group group = groupDao.findById(groupId);
            int memberCount;
            try {
                memberCount = directoryService.getGroupMembersInfo(group).getGroupMembers().size();
            } catch (TooBigGroupException e) {
                logger.error("group {} ({}) is too big. can't count void date properly",
                        group.getExternalId(), groupId);
                memberCount = 0;
            }
            groupMembersCount.put(groupId, memberCount);
            memberCountO = Option.of(groupMembersCount.getTs(groupId));
        }
        return memberCountO.get();
    }

    private BigDecimal getMonthPricePerMillisecond(DateTime month, BigDecimal pricePerMonth) {
        BigDecimal millisecondsInMonth = DateTimeUtils.getMillisecondsInMonth(month.toLocalDate());
        return pricePerMonth.divide(millisecondsInMonth, 20, RoundingMode.HALF_UP);
    }

    private BigDecimal getLeftMilliseconds(Instant now, Instant to) {
        return BigDecimal.valueOf(to.getMillis() - now.getMillis());
    }

    @Getter
    @ToString
    @AllArgsConstructor
    public static class PricePeriod {
        Instant from;
        Instant to;
        BigDecimal price;
    }
}
