package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;

import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.balanceclient.model.response.GetPartnerBalanceContractResponseItem;

import static java.math.BigDecimal.ZERO;

@Getter
@Setter
@JsonInclude(JsonInclude.Include.NON_NULL)
@ToString
public class ClientBalanceInfo {
    private Long clientId;
    private Currency currency;
    private BigDecimal incomeSum;
    private BigDecimal actSum;
    private BigDecimal transactionsSum;
    private Option<Instant> lastTransactionsDate;
    private Option<Instant> lastActDate;

    private Option<InactiveClientBalanceInfo> inactiveBalanceInfo;

    public ClientBalanceInfo(Long clientId, Currency currency, BigDecimal currentBalance) {
        this(clientId, currency);
        this.incomeSum = currentBalance;
        inactiveBalanceInfo = Option.empty();
    }

    public ClientBalanceInfo(Long clientId, Currency currency) {
        this.clientId = clientId;
        this.currency = currency;
        incomeSum = ZERO;
        actSum = ZERO;
        transactionsSum = ZERO;
        lastActDate = Option.empty();
        lastTransactionsDate = Option.empty();
        inactiveBalanceInfo = Option.empty();
    }

    public Option<Instant> getLastActDate() {
        return lastActDate
                .orElse(inactiveBalanceInfo.flatMapO(InactiveClientBalanceInfo::getLastActDate));
    }

    public void addBalanceInfo(GetPartnerBalanceContractResponseItem balance) {
        this.setIncomeSum(incomeSum.add(balance.getClientPaymentsSum()));
        this.setActSum(actSum.add(balance.getActSum()));
        this.setLastActDate(balance.getLastActDT());

    }

    @JsonInclude
    public BigDecimal getBalance() {
        return incomeSum
                .add(inactiveBalanceInfo.map(InactiveClientBalanceInfo::getBalance).orElse(ZERO))
                .add(actSum.negate())
                .add(transactionsSum.negate());
    }
}
