package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.model.TimeIntervalCondition;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransaction;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.chemodan.app.psbilling.core.util.MathUtils;
import ru.yandex.chemodan.balanceclient.model.response.ClientActInfo;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.commune.bazinga.BazingaStopExecuteException;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.COMPLETED;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.STARTED;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.STARTING;

@AllArgsConstructor
public class DistributionPlatformCalculationService {
    private static final Logger logger = LoggerFactory.getLogger(DistributionPlatformCalculationService.class);
    private static final int CLIENTS_FETCH_BATCH_SIZE = 2000;
    private static final int CALCULATION_DELETE_BATCH_SIZE = 10000;

    private final GroupDao groupDao;
    private final TaskScheduler taskScheduler;
    private final BalanceService balanceService;
    private final DistributionPlatformTransactionsDao distributionPlatformTransactionsDao;
    private final GroupServiceTransactionsDao groupServiceTransactionsDao;
    private final DistributionPlatformCalculationDao distributionPlatformCalculationDao;
    private final TransactionTemplate transactionTemplate;


    public void scheduleDistributionPlatformCalculation() {
        LocalDate calcMonth = LocalDate.now().minusMonths(1).withDayOfMonth(1); // считаем всегда деньги за прошлый
        // месяц
        TimeIntervalCondition billingInterval = TimeIntervalCondition.builder() // период, в рамках которого должны
                // быть активные услуги
                .lowerBound(new Instant(calcMonth.toDate()))
                .upperBound(new Instant(calcMonth.plusMonths(1).toDate()))
                .build();

        ListF<Group> groups = BatchFetchingUtils.collectBatchedEntities(
                (batchSize, group) -> groupDao.findGroupsWithClid(
                        batchSize, GroupType.ORGANIZATION, Option.of(billingInterval),
                        group.map(AbstractEntity::getId)),
                Function.identityF(),
                CLIENTS_FETCH_BATCH_SIZE,
                logger
        );

        distributionPlatformCalculationDao.setCalculationObsolete(calcMonth);
        clearExistingTransactions(calcMonth);
        ListF<Long> clientIds = groups.map(x -> x.getPaymentInfo().get().getClientId()).stableUnique();
        for (Long clientId : clientIds) {
            DistributionPlatformCalculationTask task = new DistributionPlatformCalculationTask(clientId, calcMonth);
            FullJobId jobId = taskScheduler.schedule(task, Instant.now().plus(Duration.standardMinutes(1)));
            distributionPlatformCalculationDao.initCalculation(calcMonth, clientId, task.getParametersTyped().getId(),
                    jobId.toSerializedString());
        }
    }

    public void calculateClient(UUID jobId, Long clientId, LocalDate calcMonth) {
        transactionTemplate.execute(status -> {
            changeTaskStatus(jobId, clientId, calcMonth, STARTING, STARTED);

            Instant from = new Instant(calcMonth.withDayOfMonth(1).toDate());
            Instant to = new Instant(calcMonth.withDayOfMonth(1).plusMonths(1).toDate());

            ListF<Group> groups = groupDao.findGroupsByPaymentInfoClient(clientId);
            logger.debug("found groups: {}", groups.map(AbstractEntity::getId));

            ListF<ClientActInfo> clientActs = balanceService.getClientActs(clientId);
            logger.debug("found acts: {}", clientActs);

            ListF<GetClientContractsResponseItem> clientContracts = balanceService.getClientContracts(clientId);
            logger.debug("found contracts: {}", clientContracts);

            ListF<DistributionPlatformTransactionsDao.InsertData> calculations = Cf.arrayList();

            for (ClientActInfo clientAct : clientActs.filter(x -> from.isBefore(x.getDt()) && to.isAfter(x.getDt())
                    && x.getPaidAmount().compareTo(x.getAmount()) >= 0)) {
                logger.debug("processing act: {}", clientAct);
                ListF<DistributionPlatformTransactionsDao.InsertData> actCalculations = Cf.arrayList();

                GetClientContractsResponseItem contract =
                        clientContracts.find(x -> x.getId().equals(clientAct.getContractId()))
                                .orElseThrow(() -> new NotFoundException(
                                        String.format("Unable to find contract %s for client %s",
                                                clientAct.getContractId(),
                                                clientId)));
                BigDecimal totalTransactionsSum = BigDecimal.ZERO;
                for (Group group : groups) {
                    ListF<GroupServiceTransaction> transactions =
                            groupServiceTransactionsDao.findGroupTransactions(calcMonth, group.getId());
                    if (transactions.length() == 0) {
                        logger.warn("no transactions found for group {} and month {}", group, calcMonth);
                        continue;
                    }
                    BigDecimal transactionsSum = MathUtils.sum(transactions.map(GroupServiceTransaction::getAmount));
                    logger.info("found {} transactions with sum {}", transactions.size(), transactionsSum);
                    if (group.getClid().isPresent()) {
                        DistributionPlatformTransactionsDao.InsertData row =
                                DistributionPlatformTransactionsDao.InsertData.builder()
                                        .actId(clientAct.getId())
                                        .groupId(group.getId())
                                        .currency(Currency.getInstance(contract.getCurrency()))
                                        .clid(group.getClid().get())
                                        .calcMonth(calcMonth)
                                        .amount(transactionsSum)
                                        .paidUserCount(transactions.map(GroupServiceTransaction::getUserSecondsCount).stream().reduce(BigDecimal.ZERO, BigDecimal::add))
                                        .build();
                        actCalculations.add(row);
                        logger.info("got row {}", row);
                    }
                    totalTransactionsSum = totalTransactionsSum.add(transactionsSum);
                }
                if (totalTransactionsSum.compareTo(clientAct.getAmount()) != 0) {
                    throw new IllegalStateException(String.format(
                            "money sum in groups transactions %s differ from act %s sum %s for client %s and " +
                                    "calcMonth %s",
                            totalTransactionsSum, clientAct.getId(), clientAct.getAmount(), clientId, calcMonth));
                }

                logger.debug("calculated for act {}: {}", clientAct, actCalculations);
                calculations.addAll(actCalculations);
            }

            logger.debug("calculated {} calculations", calculations);
            distributionPlatformTransactionsDao.batchInsert(calculations);
            distributionPlatformCalculationDao.updateStatus(jobId, STARTED, COMPLETED);
            return true;
        });
    }

    private void clearExistingTransactions(LocalDate calcMonth) {
        logger.info("Start cleaning distribution platform calculation on {}", calcMonth);

        int deleted = CALCULATION_DELETE_BATCH_SIZE;
        while (deleted == CALCULATION_DELETE_BATCH_SIZE) {
            deleted = RetryUtils.retry(logger, 3, 100, 2,
                    () -> distributionPlatformTransactionsDao.removeCalculations(calcMonth,
                            CALCULATION_DELETE_BATCH_SIZE));
        }

        logger.info("Cleaning distribution platform calculation on {} completed", calcMonth);
    }

    private void changeTaskStatus(UUID jobId, Long clientId, LocalDate calcMonth,
                                  CalculationStatus from, CalculationStatus to) {
        if (distributionPlatformCalculationDao.updateStatus(jobId, from, to)) {
            logger.info("Calculation started. jobId: {}; clientId: {}; calcMonth: {}", jobId, clientId, calcMonth);
        } else {
            String error = String.format("Unable to change calculation status from %s to %s for job %s, it's in " +
                    "unpredictable state", from, to, jobId);
            logger.error(error);
            throw new BazingaStopExecuteException(error);
        }
    }
}
