package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.util.Currency;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.balance.CorpContract;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupStatus;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.GetPartnerBalanceContractResponseItem;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static java.math.BigDecimal.ZERO;

@AllArgsConstructor
// Balance interaction logic
public class GroupBalanceService {
    private static final Logger logger = LoggerFactory.getLogger(GroupBalanceService.class);
    private final BalanceService balanceService;
    private final FeatureFlags featureFlags;

    public ListF<CorpContract> getContractBalanceItems(Long clientId, boolean withFinished) {
        boolean finishedContract = withFinished && featureFlags.getBalanceWithInactiveContract().get()
                .isEnableByClientId(clientId);

        ListF<GetClientContractsResponseItem> contractList = finishedContract
                ? balanceService.getClientContractsWithFinished(clientId)
                : balanceService.getClientContracts(clientId);

        final MapF<Long, GetClientContractsResponseItem> contracts = contractList
                .toMapMappingToKey(GetClientContractsResponseItem::getId);

        ListF<GetPartnerBalanceContractResponseItem> balances = balanceService.getContractsBalance(contracts.keys());
        logger.info("Found balances: {}", balances);
        return balances.map(b -> new CorpContract(contracts.getOrThrow(b.getContractId()), b));
    }

    /**
     * @param clientId      - Идентификатор клиента
     * @param corpContracts - Список контрактов с балансом
     * @return Баланс клиента в разрезе валюты
     */
    public MapF<Currency, ClientBalanceInfo> getBalanceAmount(Long clientId, ListF<CorpContract> corpContracts) {
        MapF<Currency, ClientBalanceInfo> result = corpContracts
                .groupBy(c -> Currency.getInstance(c.getBalance().getCurrencyCode()))
                .mapValuesWithKey((k, v) -> calcBalance(clientId, k, v));

        logger.debug("calculated balance amounts: {} for client: {} by: {}", result, clientId, corpContracts);
        return result;
    }

    /**
     * Расчет баланса по договорам.
     * <p>
     * Если нет неактивных договоров -> считаем просто баланс для нового
     * Если есть неактивные договора -> считаем баланс для старых + баланс для нового.
     * <p>
     * Суммарный Баланс - это текущий баланс на активном договоре
     * + (плюс) сумма всех старых договором (баланс договора - (минус) акт) - (минус) акты
     * <p>
     * Дата последнего акта берем из нового, если отсутствует, то самую старшую из даты последнего акта старых
     * договоров.
     *
     * @param clientId      - Идентификатор клиента
     * @param cur           - Валюта
     * @param corpContracts - Список контрактов с балансом
     * @return Суммарная информация о балансе по всем контрактам
     */
    public ClientBalanceInfo calcBalance(Long clientId, Currency cur, ListF<CorpContract> corpContracts) {
        Option<InactiveClientBalanceInfo> inactiveBalanceO = Option.empty();
        ClientBalanceInfo curBalance = new ClientBalanceInfo(clientId, cur);

        int activeContract = 0;
        for (CorpContract corpContract : corpContracts) {
            GetPartnerBalanceContractResponseItem balance = corpContract.getBalance();

            if (corpContract.getContract().isActive()) {
                activeContract++;
                curBalance.addBalanceInfo(balance);
            } else {
                inactiveBalanceO = inactiveBalanceO
                        .map(b -> b.addBalanceInfo(balance))
                        .orElse(() -> Option.of(InactiveClientBalanceInfo.byBalanceContract(balance)));
            }
        }

        if (activeContract > 1) {
            logger.warn("Too many active contact by clientId {} and currency {}", clientId, cur);
        }

        curBalance.setInactiveBalanceInfo(inactiveBalanceO);

        return curBalance;
    }

    public GroupBillingStatus calcGroupBillingStatus(Long clientId) {
        ListF<GetPartnerBalanceContractResponseItem> contractsBalance = getContractBalanceItems(clientId, false)
                .map(CorpContract::getBalance);
        return calcGroupBillingStatus(contractsBalance, clientId);
    }

    public GroupBillingStatus calcGroupBillingStatus(Long clientId,
                                                     CollectionF<GetClientContractsResponseItem> contracts) {
        ListF<GetPartnerBalanceContractResponseItem> balanceItems =
                balanceService.getContractsBalance(contracts.map(GetClientContractsResponseItem::getId));
        return calcGroupBillingStatus(balanceItems, clientId);
    }

    public GroupBillingStatus calcGroupBillingStatus(
            CollectionF<GetPartnerBalanceContractResponseItem> contractsBalance, Long clientId) {
        GroupStatus newStatus = GroupStatus.ACTIVE;
        MapF<Long, Money> contracts = Cf.hashMap();
        Option<Instant> minPaymentDeadline = Option.empty();

        for (GetPartnerBalanceContractResponseItem item : contractsBalance) {
            if (item.getFirstDebtAmount().isPresent() && item.getFirstDebtAmount().get().compareTo(ZERO) > 0) {
                logger.info("Found debt on contract {}", item);
                if (newStatus != GroupStatus.DEBT_EXISTS) {
                    newStatus = GroupStatus.PAYMENT_REQUIRED;
                }
            }
            if (item.getExpiredDebtAmount().isPresent() && item.getExpiredDebtAmount().get().compareTo(ZERO) > 0) {
                logger.info("Found expired debt on contract {}", item);
                newStatus = GroupStatus.DEBT_EXISTS;
            }

            BigDecimal actSum = item.getActSum() == null ? ZERO : item.getActSum();
            BigDecimal paymentsSum = item.getClientPaymentsSum() == null ? ZERO : item.getClientPaymentsSum();
            BigDecimal debtAmount = paymentsSum.compareTo(actSum) > 0 ? ZERO : paymentsSum.subtract(actSum);
            logger.info("Sums of contract {} of client {}: act: {},payments: {}, debt: {}",
                    item.getContractId(), clientId, actSum, paymentsSum, debtAmount);

            if (item.getFirstDebtPaymentTermDT().isPresent()
                    && (minPaymentDeadline.isEmpty() || minPaymentDeadline.get().isAfter(item.getFirstDebtPaymentTermDT().get()))) {
                minPaymentDeadline = item.getFirstDebtPaymentTermDT();
            }
            contracts.put(item.getContractId(), new Money(debtAmount.negate(),
                    Currency.getInstance(item.getCurrencyCode())));
        }

        GroupBillingStatus result = new GroupBillingStatus(newStatus, contracts, minPaymentDeadline);
        logger.info("Calculated billing status {} for client: {}", result, clientId);
        return result;
    }
}
