package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.net.URL;
import java.util.NoSuchElementException;
import java.util.UUID;
import java.util.function.Consumer;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GeneratePresignedUrlRequest;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.ResponseHeaderOverrides;
import com.google.common.base.Throwables;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.balance.CorpContract;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.exceptions.FewMoneyForPaymentException;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentGroupServiceInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentInitiationType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupServicesManager;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.GroupServiceData;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.ActionResult;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.chemodan.app.psbilling.core.util.CurrencyUtils;
import ru.yandex.chemodan.balanceclient.exception.BalanceErrorCodeException;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.chemodan.balanceclient.model.response.PayRequestResponse;
import ru.yandex.chemodan.balanceclient.model.response.RequestPaymentMethod;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

@RequiredArgsConstructor
public class GroupBillingService {
    private static final Logger logger = LoggerFactory.getLogger(GroupBillingService.class);
    private static final int UIDS_FETCH_BATCH_SIZE = 2000;
    private static final int GROUPS_FETCH_BATCH_SIZE = 2000;
    private static final int PAYMENT_REQUEST_FETCH_BATCH_SIZE = 1000;

    private final BalanceService balanceService;
    private final GroupDao groupDao;
    private final GroupServiceDao groupServiceDao;
    private final GroupServicesManager groupServicesManager;
    private final GroupProductManager groupProductManager;
    private final GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;
    private final ClientBalanceCalculator clientBalanceCalculator;
    private final ClientBalanceDao clientBalanceDao;
    private final TaskScheduler taskScheduler;
    private final GroupBalanceService groupBalanceService;
    private final GroupServiceTransactionsCalculationService groupServiceTransactionsCalculationService;
    private final CardDao cardDao;
    private final GroupsManager groupsManager;
    @SuppressWarnings("unused")
    private final FeatureFlags featureFlags;

    private final AmazonS3 s3;
    private final String bucketName;
    private final String pathInBucket;

    private final DynamicProperty<Integer> invoicesLinkLifetimeMinutes =
            new DynamicProperty<>("balance.invoices.link.lifetime.minutes", 30);
    private final DynamicProperty<Integer> checkPaymentsInterval =
            new DynamicProperty<>("ps-billing-check-init-payments-interval-minutes", 15);
    private final DynamicProperty<Integer> spreadCreateInvoicesTasksInterval =
            new DynamicProperty<>("ps-billing-spread-create-invoices-interval-minutes", 120);

    /**
     * Проверяем только привязки, которые созданы в интервале между
     * now().minus(checkCardBindingsIntervalBegin) и now().minus(checkCardBindingsIntervalEnd)
     */
    private final DynamicProperty<Integer> checkCardBindingsIntervalBegin =
            new DynamicProperty<>("ps-billing-check-card-bindings-interval-begin-minutes", 2880);
    private final DynamicProperty<Integer> checkCardBindingsIntervalEnd =
            new DynamicProperty<>("ps-billing-check-card-bindings-interval-end-minutes", 1);

    /**
     * если получили от траста id карты new_card, то ретраим в цикле retryCardBindingTimesOnNewCard раз с
     * промежутком retryCardBindingIntervalOnNewCard
     */
    private final DynamicProperty<Integer> retryCardBindingIntervalOnNewCard =
            new DynamicProperty<>("ps-billing-retry-card-binding-interval-on-new-card-milliseconds", 500);
    private final DynamicProperty<Integer> retryCardBindingTimesOnNewCard =
            new DynamicProperty<>("ps-billing-retry-card-binding-times-on-new-card", 4);

    /**
     * в случае успешной привязки карты включаем автосписание только в течение
     * canEnableAutoBillingAfterCardBindingInterval
     */
    private final DynamicProperty<Integer> canEnableAutoBillingAfterCardBindingInterval =
            new DynamicProperty<>("ps-billing-can-enable-autobilling-after-card-binding-interval-minutes", 1440);

    /**
     * если за cardBindingTimeoutForErrorStatus не получили error и success от траста, считаем, что привязка error
     */
    private final DynamicProperty<Integer> cardBindingTimeoutForErrorStatus =
            new DynamicProperty<>("ps-billing-card-binding-timeout-for-error-status-minutes", 60);

    public GroupBillingStatus actualGroupBillingStatus(Group group) {
        Validate.isTrue(group.getType() == GroupType.ORGANIZATION, "expected group type " + GroupType.ORGANIZATION);
        return group.getPaymentInfo()
                .map(paymentInfo -> {
                    Validate.isTrue(paymentInfo.getClientId() != null, "ClientId is null");
                    return updateBalanceInfo(paymentInfo, false);
                })
                .orElse(new GroupBillingStatus(GroupStatus.ACTIVE, Cf.map(), Option.empty()));
    }

    public ListF<String> createInvoices(Group group, Option<Money> money) {
        Validate.isTrue(group.getType() == GroupType.ORGANIZATION);
        BalancePaymentInfo paymentInfo = group.getPaymentInfoWithUidBackwardCompatibility()
                .orElseThrow(() -> new NoSuchElementException("there must be payment info for invoices creation"));

        MapF<Long, GetClientContractsResponseItem> contracts =
                balanceService.getClientContracts(paymentInfo.getClientId())
                        .toMapMappingToKey(GetClientContractsResponseItem::getId);

        logger.info("Found contracts {}", contracts);
        GroupBillingStatus status = groupBalanceService.calcGroupBillingStatus(
                paymentInfo.getClientId(), contracts.values());
        logger.info("Calculated billing status of group {} from balance: {}", group, status);

        ListF<String> invoices = Cf.arrayList();
        if (money.isPresent()) {
            BigDecimal amount = money.get().getAmount();
            for (Tuple2<Long, Money> entry : status.getContracts().entries()) {
                if (!entry.get2().getCurrency().equals(money.orElseThrow().getCurrency())) {
                    continue;
                }
                if (amount.compareTo(entry.get2().getAmount()) < 0) {
                    throw new FewMoneyForPaymentException(entry.get1());
                }
            }
            for (Tuple2<Long, Money> entry : status.getContracts().entries()) {
                if (!entry.get2().getCurrency().equals(money.orElseThrow().getCurrency())) {
                    continue;
                }
                GetClientContractsResponseItem contract = contracts.getOrThrow(entry.get1());
                invoices.add(createInvoice(paymentInfo, contract, new Money(amount, entry.get2().getCurrency())).getUrl());
            }
        } else {
            for (Tuple2<Long, Money> entry : status.getContractRequiredPayments().entries()) {
                GetClientContractsResponseItem contract = contracts.getOrThrow(entry.get1());
                invoices.add(createInvoice(paymentInfo, contract, entry.get2()).getUrl());
            }
        }

        return invoices;
    }

    public InvoiceFile createInvoice(BalancePaymentInfo paymentInfo,
                                     GetClientContractsResponseItem contract, Money money) {
        long paymentRequest = balanceService.createPaymentRequest(
                paymentInfo.getPassportUid(),
                paymentInfo.getClientId(),
                contract.getId().toString(),
                money.getAmount()
        );
        long invoice = balanceService.createInvoice(paymentInfo.getPassportUid(), contract, paymentRequest);
        return createSafeInvoiceUrl(balanceService.getInvoiceMdsUrl(invoice));
    }

    public void updateUidGroupsBillingStatus(PassportUid uid) {
        ListF<Group> groupsToCheck = getGroupsToCheckBillingForUid(uid);
        if (groupsToCheck.isEmpty()) {
            return;
        }
        GroupStatus aggregatedInitialGroupStatusForUid = groupsToCheck
                .map(Group::getStatus)
                .sortedByDesc(GroupStatus::getPriority)
                .first();
        logger.info("Aggregated initial group status for uid={} status={}", uid, aggregatedInitialGroupStatusForUid);
        ListF<GroupBillingStatusData> groupWithBillingStatusList = groupsToCheck
                .groupBy(g -> g.getPaymentInfo().orElseThrow(IllegalStateException::new))
                .mapEntries((paymentInfo, group) -> updateGroupsBillingStatus(paymentInfo, group, true))
                .flatten();
        if (groupWithBillingStatusList.isEmpty()) {
            return;
        }
        Instant now = Instant.now();
        triggerEmailIfNeededForStatus(uid, groupWithBillingStatusList, aggregatedInitialGroupStatusForUid,
                GroupStatus.DEBT_EXISTS, EventMailType.BECAME_FREE, now);
        triggerEmailIfNeededForStatus(uid, groupWithBillingStatusList, aggregatedInitialGroupStatusForUid,
                GroupStatus.PAYMENT_REQUIRED, EventMailType.ISSUE_AN_INVOICE, now);
    }

    public void scheduleUidsWithExpiredBillingDate() {
        BatchFetchingUtils.<PassportUid, PassportUid>fetchAndProcessBatchedEntities(
                (batchSize, uid) -> groupDao.findUidsToCheckBillingStatus(batchSize, GroupType.ORGANIZATION, uid),
                Function.identityF(),
                uid -> RetryUtils.retry(logger, 3, 500, 2,
                        () -> taskScheduler.schedule(new CheckGroupBillingStatusTask(uid))),
                UIDS_FETCH_BATCH_SIZE,
                logger
        );
    }

    public InvoiceFile createSafeInvoiceUrl(String balanceUrl) {
        URL url = UrlUtils.url(balanceUrl);
        String[] split = StringUtils.split(url.getPath(), "/");
        Validate.isTrue(split.length == 2);
        String fromBucket = split[0];
        String fromFile = split[1];

        String key = StringUtils.isBlank(pathInBucket) ? "" : StringUtils.appendIfMissing(pathInBucket, "/");
        key += UUID.randomUUID().toString();

        try {
            s3.copyObject(fromBucket, fromFile, bucketName, key);
        } catch (AmazonS3Exception e) {
            logger.warn("Unable to copy file from balance bucket to our bucket, trying to download and upload file", e);
            //try to upload file
            try {
                ByteArrayInputStream fileContent =
                        new ByteArrayInputStream(IOUtils.toByteArray(URI.create(balanceUrl)));
                s3.putObject(new PutObjectRequest(bucketName, key, fileContent, null));
            } catch (IOException ex) {
                logger.error("failed to upload file", ex);
                throw Throwables.propagate(ex);
            }
        }

        ResponseHeaderOverrides responseHeaderOverrides = new ResponseHeaderOverrides();
        responseHeaderOverrides.setContentDisposition("attachment; filename=" + fromFile);

        URL presignedUrl = s3.generatePresignedUrl(new GeneratePresignedUrlRequest(bucketName, key)
                .withExpiration(
                        Instant.now().plus(Duration.standardMinutes(invoicesLinkLifetimeMinutes.get())).toDate())
                .withResponseHeaders(responseHeaderOverrides)
        );

        return new InvoiceFile(fromFile, presignedUrl.toString());
    }

    public TrustPaymentFormData getTrustPaymentFormUrl(PassportUid uid, Group group, String redirectUrl,
                                                       Option<String> theme, Option<String> title,
                                                       Option<String> groupProductCode,
                                                       Option<Money> money, Option<String> payload) {
        BalancePaymentInfo paymentInfo = group.getPaymentInfo().orElseThrow(IllegalStateException::new);
        ListF<GetClientContractsResponseItem> contracts = balanceService.getClientContracts(paymentInfo.getClientId());

        GroupBillingStatus status = groupBalanceService.calcGroupBillingStatus(paymentInfo.getClientId());
        MapF<Long, Money> contractsForPayment = money.isPresent() ?
                status.getContracts().filterValues(x -> x.getCurrency().equals(money.orElseThrow().getCurrency())) :
                status.getContractRequiredPayments();

        if (contractsForPayment.isEmpty()) {
            throw new IllegalStateException("No contracts for payment");
        }
        if (contractsForPayment.size() > 1) {
            throw new IllegalStateException(String.format("Contracts for payments more than 1. Total count: %s",
                    contractsForPayment.size()));
        }
        if (money.isPresent() && money.get().getAmount().compareTo(contractsForPayment.entries().first().get2().getAmount()) < 0) {
            throw new FewMoneyForPaymentException(contractsForPayment.entries().first().get1());
        }
        Tuple2<Long, Money> contractPaymentData = contractsForPayment.entries().first();
        Money moneyForPayment = money.orElse(contractPaymentData._2);

        Long contractId = contractPaymentData._1;
        String requestId = String.valueOf(balanceService.createPaymentRequest(uid, paymentInfo.getClientId(),
                contracts.find(contract -> contractId.equals(contract.getId())).first().getId().toString(),
                moneyForPayment.getAmount()));
        String contractIdStringValue = contractId.toString();
        RequestPaymentMethod paymentMethod =
                balanceService.getRequestPaymentMethodForUnboundCard(uid, requestId, contractIdStringValue)
                        .orElseThrow(() -> new IllegalStateException("No card payment method"));
        PayRequestResponse payRequest =
                balanceService.createPayRequestByTrustForm(uid, requestId, contractIdStringValue,
                        paymentMethod.getCurrency(), redirectUrl, theme, title, payload);

        groupTrustPaymentRequestDao.insert(GroupTrustPaymentRequestDao.InsertData.builder()
                .clientId(paymentInfo.getClientId())
                .requestId(requestId)
                .status(PaymentRequestStatus.INIT)
                .transactionId(Option.of(payRequest.getPurchaseToken()))
                .operatorUid(uid.toString())
                .paymentInitiationType(PaymentInitiationType.USER)
                .money(moneyForPayment)
                .build());

        if (featureFlags.getB2bAdminAbandonedPaymentEmailEnabled().isEnabledForUid(uid) &&
                groupProductCode.isPresent()) {
            GroupProduct groupProduct = groupProductManager.findProduct(groupProductCode.get());
            taskScheduler.scheduleB2bAdminAbandonedPaymentTask(uid, paymentInfo.getClientId(),
                    groupProduct.getUserProductId(), group.getId());
        }
        return new TrustPaymentFormData(payRequest.getPaymentUrl(), requestId);
    }

    public void chargePayment(PassportUid uid, long clientId, Money payment, CardEntity card,
                                            Option<Double> paymentPeriodCoefficient,
                                            Option<GroupTrustPaymentGroupServiceInfo> servicesToResurrect) {
        logger.info("charge automatic payment: uid {}, clientId {}, card {}, payment {}, paymentPeriodCoefficient {}",
                uid, clientId, payment, card, paymentPeriodCoefficient);

        String currencyCode = payment.getCurrency().getCurrencyCode();
        GetClientContractsResponseItem contract =
                balanceService.getActiveContract(clientId)
                        .filter(x -> CurrencyUtils.normalizedEquals(x.getCurrency(), currencyCode))
                        .orElseThrow(() -> new IllegalStateException(
                                String.format("no active contract for client %s and currency %s",
                                        clientId, payment.getCurrency())));
        String contractId = contract.getId().toString();

        String requestId = String.valueOf(balanceService.createPaymentRequest(
                uid, clientId, contractId, payment.getAmount()));

        RequestPaymentMethod paymentMethod = balanceService.getBoundCards(uid, requestId, contractId)
                .filter(x -> x.getPaymentMethodId().isSome(card.getExternalId()))
                .firstO()
                .orElseThrow(() -> new IllegalStateException("Card not available for payment"));

        GroupTrustPaymentRequest groupTrustPaymentRequest = groupTrustPaymentRequestDao.insert(
                GroupTrustPaymentRequestDao.InsertData.builder()
                        .clientId(clientId)
                        .requestId(requestId)
                        .status(PaymentRequestStatus.INIT)
                        .transactionId(Option.empty())
                        .operatorUid(uid.toString())
                        .paymentInitiationType(PaymentInitiationType.AUTO)
                        .cardId(Option.of(card.getId()))
                        .paymentCoefficient(paymentPeriodCoefficient)
                        .groupServicesInfo(servicesToResurrect)
                        .money(payment)
                        .build()
        );

        PayRequestResponse payResponse = balanceService.createAutomaticPayRequest(
                uid,
                requestId,
                contractId,
                paymentMethod.getPaymentMethodId().get(),
                paymentMethod.getCurrency()
        );

        groupTrustPaymentRequestDao.updateTransactionIdIfNull(
                groupTrustPaymentRequest.getId(),
                payResponse.getPurchaseToken()
        );
    }

    public void scheduleToCheckOldInitPaymentRequests() {
        Instant olderThan = Instant.now().minus(Duration.standardMinutes(checkPaymentsInterval.get()));
        Consumer<GroupTrustPaymentRequest> consumer = requestsToCheck -> Option.of(requestsToCheck)
                .map(paymentRequest -> new CheckPaymentRequestStatusTask.Parameters(paymentRequest.getRequestId()))
                .map(CheckPaymentRequestStatusTask::new)
                .map(taskScheduler::schedule);
        BatchFetchingUtils.fetchAndProcessBatchedEntities(
                (batchSize, id) -> groupTrustPaymentRequestDao.findInitPaymentsOlderThan(olderThan, batchSize, id),
                GroupTrustPaymentRequest::getId,
                consumer,
                PAYMENT_REQUEST_FETCH_BATCH_SIZE,
                logger
        );
    }

    private GroupBillingStatus updateBalanceInfo(BalancePaymentInfo paymentInfo, boolean calcTodayTransactions) {
        Long clientId = paymentInfo.getClientId();
        ListF<CorpContract> balanceItems = groupBalanceService.getContractBalanceItems(clientId, true);

        if (calcTodayTransactions) {
            groupServiceTransactionsCalculationService.calculateClientTransactions(clientId, LocalDate.now());
        }

        clientBalanceCalculator.updateClientBalance(clientId, balanceItems);

        return groupBalanceService.calcGroupBillingStatus(
                balanceItems
                        .filter(c -> c.getContract().isActive())
                        .map(CorpContract::getBalance),
                clientId);
    }

    private Instant getCreateInvoiceTaskScheduleTime(Instant now) {
        return new Instant(now.getMillis() +
                Random2.R.nextLong(Duration.standardMinutes(spreadCreateInvoicesTasksInterval.get()).getMillis()));
    }

    private ListF<Group> getGroupsToCheckBillingForUid(PassportUid uid) {
        return BatchFetchingUtils.collectBatchedEntities(
                (batchSize, groupId) -> groupDao
                        .getGroupsToCheckForUid(uid, batchSize, GroupType.ORGANIZATION, groupId),
                Group::getId,
                GROUPS_FETCH_BATCH_SIZE,
                logger
        );
    }

    public ListF<GroupBillingStatusData> updateGroupsBillingStatus(BalancePaymentInfo paymentInfo,
                                                                    ListF<Group> groups,
                                                                    boolean calcTodayTransactions) {
        logger.info("Checking status in balance for payment_info {}", paymentInfo);
        try {
            GroupBillingStatus newStatus = updateBalanceInfo(paymentInfo, calcTodayTransactions);
            return groups.filterMap(g -> updateGroupBillingStatus(newStatus, g));
        } catch (BalanceErrorCodeException ex) {
            // раз в некоторое время Баланс переналивает базу тестинга базой из прода,
            // и клиенты, которые были созданы в тестинге, умирают, и у нас начинают массово падать таски всякие.
            // Поэтому не для прода можно выключать все услуги
            if (EnvironmentType.getActive() != EnvironmentType.PRODUCTION
                    && ex.getCodes().containsTs(BalanceErrorCodeException.BalanceErrorCode.CLIENT_NOT_FOUND.name())) {
                logger.warn("client {} not found in Balance. Will disable all services", paymentInfo.getClientId());
                groups.forEach(g ->
                        groupServicesManager.find(g.getId(), Target.ENABLED)
                                .forEach(groupServicesManager::disableGroupService));
                return Cf.list();
            }
            throw ex;
        }
    }

    private Option<GroupBillingStatusData> updateGroupBillingStatus(GroupBillingStatus newStatus, Group group) {
        logger.info("Updating billing status for group {} from oldStatus {} with newStatus {}", group,
                group.getStatus(), newStatus);
        groupDao.updateStatus(group.getId(), newStatus.getStatus());

        ListF<GroupService> groupServices = groupServiceDao.find(group.getId(), Target.ENABLED);
        for (GroupService groupService : groupServices) {
            updateBdate(group, newStatus, groupService);
        }

        if (newStatus.getStatus().getPriority() > group.getStatus().getPriority()) {
            return Option.of(new GroupBillingStatusData(group, Cf.list(), newStatus.getStatus()));
        }
        return Option.empty();
    }

    private void updateBdate(Group group, GroupBillingStatus newStatus, GroupService groupService) {
        GroupProduct groupProduct = groupProductManager.findById(groupService.getGroupProductId());
        if (groupProduct.isFree() || groupService.isSkipTransactionsExport()) {
            logger.info("setting bdate for free service {} to null", groupService);
            groupServiceDao.updateNextBillingDate(groupService.getId(), Option.empty());
            return;
        }

        if (groupProduct.getPaymentType().equals(GroupPaymentType.POSTPAID)) {
            updatePostpaidServiceBdate(group, newStatus, groupService);
        }

        if (groupProduct.getPaymentType().equals(GroupPaymentType.PREPAID)) {
            updatePrepaidServiceBdate(group, newStatus, groupService, groupProduct);
        }
    }

    private void updatePostpaidServiceBdate(Group group, GroupBillingStatus newStatus, GroupService groupService) {
        if (newStatus.getStatus().equals(GroupStatus.DEBT_EXISTS)) {
            logger.info("Switching off group service {} because debt exist", groupService);
            groupServicesManager.disableGroupService(groupService);
            return;
        }

        if (newStatus.getStatus().equals(GroupStatus.PAYMENT_REQUIRED)) {
            Instant nextCheckDate = newStatus.getFirstDebtPaymentDeadline()
                    .orElse(Instant.now().plus(Duration.standardDays(1)));
            groupServiceDao.updateNextBillingDate(groupService.getId(), Option.of(nextCheckDate));
            return;
        }

        // Should take last act date from Balance plus 1 month, but this field not yet provided
        // So we can shift date at grace period - we can be sure that debt will not exist in now+grace tim
        Instant nextCheckDate = Instant.now().plus(group.getGracePeriod().orElse(Duration.standardDays(1)));
        groupServiceDao.updateNextBillingDate(groupService.getId(), Option.of(nextCheckDate));
    }

    private void updatePrepaidServiceBdate(Group group, GroupBillingStatus newStatus, GroupService groupService,
                                           GroupProduct groupProduct) {
        if (newStatus.getStatus().equals(GroupStatus.DEBT_EXISTS)) {
            logger.info("Switching off group service {} because debt exist", groupService);
            groupServicesManager.disableGroupService(groupService);
            return;
        }

        Option<ClientBalanceEntity> clientBalanceO = clientBalanceDao.findByGroupService(groupService);
        ClientBalanceEntity clientBalance;
        if (!clientBalanceO.isPresent()) {
            // not really possible branch
            logger.info("balance for group service {} not found", groupService);
            Option<Long> clientIdO = group.getPaymentInfo().map(BalancePaymentInfo::getClientId);
            if (!clientIdO.isPresent()) {
                throw new IllegalStateException(String.format("no payment info for group %s", group));
            }
            clientBalance = clientBalanceCalculator.updateClientBalance(clientIdO.get(),
                    groupProduct.getPriceCurrency());
        } else {
            clientBalance = clientBalanceO.get();
        }
        Option<Instant> nextBdate = clientBalance.getBalanceVoidAt();

        if (newStatus.getStatus().equals(GroupStatus.PAYMENT_REQUIRED) && newStatus.getFirstDebtPaymentDeadline().isPresent()) {
            if (!nextBdate.isPresent() || nextBdate.get().isAfter(newStatus.getFirstDebtPaymentDeadline().get())) {
                nextBdate = newStatus.getFirstDebtPaymentDeadline();
            }
        }

        groupServiceDao.updateNextBillingDate(groupService.getId(), nextBdate);
        if (!nextBdate.isPresent()) {
            logger.warn("no bdate for prepaid product of service {}", groupService);
            return;
        }

        if (!nextBdate.get().isAfterNow()) {
            logger.info("disabling prepaid service {} cause bdate is in past", groupService);
            groupServicesManager.disableGroupService(groupService);
            groupServicesManager.scheduleBalanceExhaustedEmails(groupService, group);
        }
    }

    private void triggerEmailIfNeededForStatus(PassportUid uid,
                                               ListF<GroupBillingStatusData> groupBillingStatusDataList,
                                               GroupStatus initialUidGroupBillingStatus,
                                               GroupStatus newGroupBillingStatus,
                                               EventMailType eventMailType,
                                               Instant now) {
        if (!(initialUidGroupBillingStatus.getPriority() < newGroupBillingStatus.getPriority())) {
            logger.info("Not trigger an email cause of priority oldValue={} newValue={}", initialUidGroupBillingStatus,
                    newGroupBillingStatus);
            return;
        }
        ListF<GroupBillingStatusData> groupsWithNewBillingStatus = groupBillingStatusDataList
                .filter(groupBillingStatusData -> newGroupBillingStatus
                        .equals(groupBillingStatusData.getGroupStatus()));
        if (groupsWithNewBillingStatus.isEmpty()) {
            logger.info("No groups with status={} allData={}", newGroupBillingStatus, groupBillingStatusDataList);
            return;
        }
        logger.info("Triggering the {} email for uid={}", eventMailType, uid);
        MailContext mailContext = MailContext.builder()
                .to(uid)
                .groupIds(
                        groupsWithNewBillingStatus.map(GroupBillingStatusData::getGroup)
                                .map(Group::getId)
                                .map(UUID::toString)
                )
                .groupServices(groupsWithNewBillingStatus.flatMap(GroupBillingStatusData::getGroupServices))
                .userServiceId(Option.empty())
                .language(Option.empty())
                .build();
        taskScheduler.scheduleTransactionalEmailTask(eventMailType, mailContext,
                getCreateInvoiceTaskScheduleTime(now));
    }

    @Transactional
    public ActionResult setAutoBillingOff(BalancePaymentInfo paymentInfo) {
        setAutoBillingForClient(paymentInfo.getClientId(), false);
        return ActionResult.success("auto billing has been disabled");
    }

    public ActionResult setAutoBillingOn(BalancePaymentInfo paymentInfo) {
        return setAutoBillingOn(paymentInfo, false);
    }

    public ActionResult setAutoBillingOn(BalancePaymentInfo paymentInfo, boolean skipCardCheck) {
        logger.info("try to set auto billing on for payment info: {}; skip card check: {}", paymentInfo, skipCardCheck);
        PassportUid uid = paymentInfo.getPassportUid();
        long clientId = paymentInfo.getClientId();

        if (paymentInfo.isB2bAutoBillingEnabled()) {
            return ActionResult.success("auto billing has been already enabled");
        }

        if (groupsManager.hasActivePostpaidGroupServices(uid)) {
            return ActionResult.fail("Has active postpaid services",
                    String.format("Can't enable b2b auto billing because client %s has postpaid services", paymentInfo),
                    "has_active_postpaid_services");
        }

        if (!skipCardCheck) {
            Option<CardEntity> primaryCard = cardDao.findB2BPrimary(uid);
            if (primaryCard.isEmpty() || primaryCard.get().getStatus() != CardStatus.ACTIVE) {
                return ActionResult.fail("No primary card",
                        String.format("Can't enable b2b auto billing because user %s has no active primary card", uid),
                        "no_primary_card");
            }
        }

        setAutoBillingForClient(clientId, true);
        return ActionResult.success("auto billing has been enabled");
    }

    private void setAutoBillingForClient(Long clientId, boolean enableAutoBilling) {
        logger.info("{} auto billing for clientId {}", enableAutoBilling ? "enable" : "disable", clientId);
        groupDao.setAutoBillingForClient(clientId, enableAutoBilling);
    }
    @Data
    public static class TrustPaymentFormData {
        private final String url;
        private final String paymentRequestId;
    }

    @Data
    private static class GroupBillingStatusData {
        private final Group group;
        private final ListF<GroupServiceData> groupServices;
        private final GroupStatus groupStatus;
    }
}


