package ru.yandex.chemodan.app.psbilling.core.billing.groups;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.UUID;

import com.google.common.util.concurrent.Runnables;
import lombok.AllArgsConstructor;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.groupservices.GroupServiceTransactionsExportService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks.UpdateClientTransactionsBalanceTask;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransaction;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransactionCalculation;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.commune.bazinga.BazingaStopExecuteException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.bolts.function.Function.identityF;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.COMPLETED;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.STARTED;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.STARTING;

@AllArgsConstructor
public class GroupServiceTransactionsCalculationService {
    private static final Logger logger = LoggerFactory.getLogger(GroupServiceTransactionsCalculationService.class);
    private static final int CALCULATION_BATCH_SIZE = 500;
    private static final int CALCULATION_DELETE_BATCH_SIZE = 10000;
    private static final LocalDate BILLING_START_DATE = LocalDate.parse("2020-02-03");

    private final BazingaTaskManager bazingaTaskManager;
    private final GroupServiceDao groupServiceDao;
    private final GroupProductDao groupProductDao;
    private final GroupServiceTransactionsDao groupServiceTransactionsDao;
    private final GroupServiceTransactionCalculationDao groupServiceTransactionCalculationDao;
    private final GroupServiceTransactionsExportService groupServiceTransactionsExportService;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;
    private final TransactionTemplate transactionTemplate;
    private final TaskScheduler taskScheduler;

    public void calculateGroupServicesTransactions(LocalDate date) {
        logger.info("Start calculation on {}", date);
        Option<GroupServiceDao.Page> currentPage = nextPageWithRetries(date, Option.empty());
        logger.info("Next page to calculate: {} on {}", currentPage, date);

        while (currentPage.isPresent()) {
            GroupServiceDao.Page page = currentPage.get();

            MapF<GroupService, ListF<Interval>> usages =
                    withRetries(() -> groupServiceDao.getEnabledUserServicesInDayPaidIntervals(date, page));
            logger.info("found services with billing intervals for page {} on date {}: {}", page, date,
                    usages.keySet());

            calculateTransactions(date, usages);

            currentPage = nextPageWithRetries(date, currentPage);
            logger.info("Next page to calculate: {} on {}", currentPage, date);
        }

        logger.info("Calculation completed, changing status to completed");
        boolean updated = withRetries(
                () -> groupServiceTransactionCalculationDao.updateStatus(date, STARTED, COMPLETED));
        if (updated) {
            logger.info("Calculation on {} completed. Scheduling update client balance task", date);
            taskScheduler.schedule(new UpdateClientTransactionsBalanceTask(date));
        } else {
            logger.error("Unable to change calculation status, it's in unpredictable state");
            throw new BazingaStopExecuteException("Unable to change calculation status, it's in unpredictable state");
        }
    }

    public void calculateClientTransactions(Long clientId, LocalDate date) {
        logger.info("Start calculation for client {} on date {}", clientId, date);
        ListF<GroupService> groupServices = groupServiceDao.findActiveServicesOnDayForClient(date, clientId);
        MapF<GroupService, ListF<Interval>> usages =
                withRetries(() -> groupServiceDao.getEnabledUserServicesInDayPaidIntervals(date, groupServices));
        calculateTransactions(date, usages);
        logger.info("Calculation completed");
    }

    public void scheduleNotCalculated() {
        LocalDate twoMonthsAgo = LocalDate.now().minusMonths(2);
        LocalDate startOfCheck = twoMonthsAgo.isBefore(BILLING_START_DATE) ? BILLING_START_DATE : twoMonthsAgo;

        LocalDate now = LocalDate.now();
        LocalDate dateToCheck = startOfCheck;
        logger.info("start checking calculations from(>=) {} to(<) {}", dateToCheck, now);
        while (dateToCheck.isBefore(now)) {
            tryStartCalculationAsync(dateToCheck);

            dateToCheck = dateToCheck.plusDays(1);
        }
    }

    public String scheduleRecalculation(LocalDate startDate, LocalDate endDate, boolean export) {
        FullJobId jobId = bazingaTaskManager.schedule(new ManualRecalculationTask(startDate, endDate, export));
        return jobId.toString();
    }

    public Option<OnetimeJob> getRecalculationJob(String jobId) {
        return bazingaTaskManager.getOnetimeJob(FullJobId.parse(jobId));
    }

    public void recalculate(LocalDate startDate, LocalDate endDate, boolean export) throws Exception {
        LocalDate dateToStart = startDate;

        while (dateToStart.isBefore(endDate) || dateToStart.isEqual(endDate)) {
            clearExistingTransactionsOnDay(dateToStart);
            performSyncCalculation(dateToStart, true);
            dateToStart = dateToStart.plusDays(1);
        }

        if (export) {
            groupServiceTransactionsExportService.forceExportLatestCalulcations();
        }
    }


    private void calculateTransactions(LocalDate date, MapF<GroupService, ListF<Interval>> usages) {
        Instant now = Instant.now();
        if (!usages.isEmpty()) {
            MapF<UUID, GroupProductEntity> products = withRetries(
                    () -> groupProductDao.findByIds(usages.keys().map(GroupService::getGroupProductId).unique())
                            .toMap(GroupProductEntity::getId, identityF())
            );
            MapF<UUID, ListF<GroupServicePriceOverride>> pricesOverrides = withRetries(() ->
                    groupServicePriceOverrideDao.findByGroupServices(usages.keySet().map(GroupService::getId)));

            withRetriesV(() -> groupServiceTransactionsDao.batchInsert(usages.mapEntries(
                    (groupService, intervals) -> {
                        GroupProductEntity groupProductEntity =
                                products.getOrThrow(groupService.getGroupProductId());
                        ListF<GroupServicePriceOverride> priceOverrides =
                                pricesOverrides.getO(groupService.getId()).orElse(Cf.list());
                        CalculationResult calculationResult =
                                calculateAmountOnDate(groupService, groupProductEntity, priceOverrides, intervals
                                        , date);
                        return GroupServiceTransaction.builder()
                                .billingDate(date)
                                .groupServiceId(groupService.getId())
                                .currency(groupProductEntity.getPriceCurrency())
                                .amount(calculationResult.amount)
                                .userSecondsCount(calculationResult.userSeconds)
                                .calculatedAt(calculationResult.upperTimeBound
                                        .map(limit -> limit.isAfter(now) ? now : limit).orElse(now))
                                .build();
                    }).filter(gst -> gst.getAmount().compareTo(BigDecimal.ZERO) > 0)));
        }
    }

    private Option<GroupServiceDao.Page> nextPageWithRetries(LocalDate date, Option<GroupServiceDao.Page> prevPage) {
        return withRetries(
                () -> groupServiceDao.nextPageOfActiveServicesOnDay(date, prevPage.map(GroupServiceDao.Page::getMax),
                        CALCULATION_BATCH_SIZE)
        );
    }

    private static CalculationResult calculateAmountOnDate(GroupService groupService,
                                                           GroupProductEntity groupProductEntity,
                                                           ListF<GroupServicePriceOverride> priceOverrides,
                                                           ListF<Interval> intervalsOfUsage,
                                                           LocalDate billingDate) {
        BigDecimal millisecondsInMonth = DateTimeUtils.getMillisecondsInMonth(billingDate);
        priceOverrides = priceOverrides.sortedBy(GroupServicePriceOverride::getStartDate);

        BigDecimal amount = BigDecimal.ZERO;
        BigDecimal userMilliSecondsCountTotal = BigDecimal.ZERO;

        Option<Instant> usageLimitTimeO = Option.empty();
        if (groupProductEntity.getPaymentType().equals(GroupPaymentType.POSTPAID) && groupService.getTarget().equals(Target.DISABLED)) {
            usageLimitTimeO = Option.of(groupService.getTargetUpdatedAt());
        }
        if (groupProductEntity.getPaymentType().equals(GroupPaymentType.PREPAID)) {
            usageLimitTimeO = groupService.getNextBillingDate();
        }

        for (Interval interval : intervalsOfUsage) {
            ListF<ClientBalanceCalculator.PricePeriod> pricesInterval =
                    ClientBalanceCalculator.slitByPricesInterval(interval, priceOverrides,
                            groupProductEntity.getPricePerUserInMonth());
            for (ClientBalanceCalculator.PricePeriod intervalWithPrice : pricesInterval) {
                Interval curInterval = new Interval(intervalWithPrice.from, intervalWithPrice.to);
                BigDecimal price = intervalWithPrice.price;

                if (usageLimitTimeO.isPresent()) {
                    logger.info("time limit for calculation is {}", usageLimitTimeO.get());
                    if (usageLimitTimeO.get().compareTo(curInterval.getStart()) <= 0) {
                        continue;
                    }
                    if (usageLimitTimeO.get().compareTo(curInterval.getEnd()) < 0) {
                        curInterval = curInterval.withEnd(usageLimitTimeO.get().toDateTime());
                    }
                }

                BigDecimal userMilliSecondsCount = new BigDecimal(curInterval.toDurationMillis());
                userMilliSecondsCountTotal = userMilliSecondsCountTotal.add(userMilliSecondsCount);
                amount = amount.add(price.multiply(userMilliSecondsCount));
            }
        }

        amount = amount.divide(millisecondsInMonth, 2, RoundingMode.HALF_UP);
        logger.info("Calculated {} amount, userMilliSecondsCountTotal {}, pricePerUserInMonth {}, billing date {}",
                amount, userMilliSecondsCountTotal, groupProductEntity.getPricePerUserInMonth(), billingDate);

        BigDecimal userSecondsCountTotal = userMilliSecondsCountTotal.divide(BigDecimal.valueOf(1000), 0, RoundingMode.DOWN);
        return new CalculationResult(amount, userSecondsCountTotal, usageLimitTimeO);
    }

    private void tryStartCalculationAsync(LocalDate billingDate) {
        tryStartCalculation(billingDate, false, () -> {
            FullJobId jobId = bazingaTaskManager.schedule(new GroupServiceTransactionsCalculationTask(billingDate));
            withRetriesV(() -> groupServiceTransactionCalculationDao
                    .saveBazingaJobId(billingDate, jobId.toSerializedString()));
        });
    }

    public void performSyncCalculation(LocalDate billingDate, boolean force) {
        if (tryStartCalculation(billingDate, force, Runnables::doNothing)) {
            calculateGroupServicesTransactions(billingDate);
        }
    }

    private boolean tryStartCalculation(LocalDate billingDate, boolean force, Function0V callback) {
        groupServiceTransactionCalculationDao.insertIfAbsent(billingDate);

        return transactionTemplate.execute(status -> {
            GroupServiceTransactionCalculation calculation = groupServiceTransactionCalculationDao.lock(billingDate);
            logger.info("DB calculation on day {}: {}", billingDate, calculation);
            if (calculation.getStatus() == STARTING || (force && calculation.getStatus() == COMPLETED)) {
                logger.info("Starting {} calculation on {}", force ? "force" : "", billingDate);
                groupServiceTransactionCalculationDao.updateStatus(billingDate, calculation.getStatus(), STARTED);
                callback.apply();
                return true;
            } else {
                logger.info("No {} calculation on {} needed", force ? "force" : "", billingDate);
                return false;
            }
        });
    }


    public void clearExistingTransactionsOnDay(LocalDate billingDate) {
        logger.info("Start cleaning calculation on {}", billingDate);

        int deleted = CALCULATION_DELETE_BATCH_SIZE;
        while (deleted == CALCULATION_DELETE_BATCH_SIZE) {
            deleted = RetryUtils.retry(logger, 3, 100, 2,
                    () -> groupServiceTransactionsDao.removeCalculations(billingDate, CALCULATION_DELETE_BATCH_SIZE));
        }

        logger.info("Cleaning calculation on {} completed", billingDate);
    }


    public void scheduleRecalculation(LocalDate startDate, LocalDate endDate) {
        Validate.isTrue(endDate.compareTo(startDate) >= 0);

        LocalDate dateToStart = startDate;
        while (dateToStart.isBefore(endDate) || dateToStart.isEqual(endDate)) {
            tryStartCalculationAsync(dateToStart);
            dateToStart = dateToStart.plusDays(1);
        }
    }

    private static <T> T withRetries(Function0<T> action) {
        return RetryUtils.retry(logger, 3, 100, 2, action);
    }

    private static void withRetriesV(Function0V action) {
        RetryUtils.retry(logger, 3, 100, 2, () -> {
            action.apply();
            return null;
        });
    }

    @AllArgsConstructor
    private static class CalculationResult {
        BigDecimal amount;
        BigDecimal userSeconds;
        Option<Instant> upperTimeBound;
    }
}
