package ru.yandex.chemodan.app.psbilling.core.billing.groups.export;


import lombok.Getter;
import lombok.SneakyThrows;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public abstract class BaseExportService<TExportRow extends YtExportRow, TYtOperations extends YtOperations<TExportRow>> {
    private static final Logger logger = LoggerFactory.getLogger(BaseExportService.class);
    private static final DateTimeFormatter DATE_FORMATTER = ISODateTimeFormat.date();

    protected final BazingaTaskManager bazingaTaskManager;
    protected final DynamicProperty<Integer> exportsToKeep =
            new DynamicProperty<>("transactions.exports.keep.count", 30);

    @Getter
    protected final TYtOperations primaryYtOperations;

    @Getter
    protected final TYtOperations secondaryYtOperations;

    public BaseExportService(BazingaTaskManager bazingaTaskManager,
                             TYtOperations primaryYtOperations,
                             TYtOperations secondaryYtOperations) {
        this.bazingaTaskManager = bazingaTaskManager;
        this.primaryYtOperations = primaryYtOperations;
        this.secondaryYtOperations = secondaryYtOperations;
    }

    public abstract void scheduleExport();

    @SneakyThrows
    public void export() {
        doForTwoClusters("Export", operations -> doExport(operations, false));
    }

    @SneakyThrows
    public void removeOldExports() {
        doForTwoClusters("Remove old exports", this::removeOldExports);
    }

    public Option<Instant> getLastPrimaryClusterExportTime() {
        return primaryYtOperations.getCurrentExportInstant();
    }

    public Option<Instant> getLastSecondaryClusterExportTime() {
        return secondaryYtOperations.getCurrentExportInstant();
    }

    protected abstract void removeOldExports(TYtOperations exporter);

    protected abstract void doExport(TYtOperations exporter, boolean force);

    protected void doForTwoClusters(String operationName, Function1V<TYtOperations> callback) throws Exception {
        Option<Exception> finalException =
                doForCluster(primaryYtOperations, operationName, "primary", callback);
        Option<Exception> exception =
                doForCluster(secondaryYtOperations, operationName, "secondary", callback);

        finalException = finalException.orElse(exception);
        if (finalException.isPresent()) {
            throw finalException.get();
        }
    }

    private Option<Exception> doForCluster(TYtOperations operations, String operationName, String clusterName,
                                           Function1V<TYtOperations> callback) {
        if (!operations.isEnabled()) {
            logger.info("{} to {} YT cluster disabled", operationName, clusterName);
            return Option.empty();
        }

        try {
            callback.apply(operations);
        } catch (Exception e) {
            logger.error("{} to {} YT cluster failed: {}", operationName, clusterName, e.getMessage(), e);
            return Option.of(e);
        }
        return Option.empty();
    }

    protected static String buildTablePathForDate(String rootPath, LocalDate date) {
        return rootPath + "/" + DATE_FORMATTER.print(date);
    }
}
