package ru.yandex.chemodan.app.psbilling.core.billing.groups.export;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.groupservices.YtOperationsGroupTransactionsImpl;
import ru.yandex.chemodan.app.psbilling.core.config.YtExportSettings;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public abstract class BaseYtOperations<TExportRow extends YtExportRow> implements YtOperations<TExportRow> {
    private static final Logger logger = LoggerFactory.getLogger(YtOperationsGroupTransactionsImpl.class);

    protected static final DateTimeFormatter DATE_FORMATTER = ISODateTimeFormat.date();
    protected static final DateTimeFormatter MONTH_FORMATTER = ISODateTimeFormat.yearMonth();
    protected static final DateTimeFormatter ISO_FORMAT = ISODateTimeFormat.dateTime();
    protected static final Duration TRANSACTION_MAX_DURATION = Duration.ofMinutes(15);

    protected final YtExportSettings settings;
    protected final Function0<Integer> batchSizeProvider;

    public boolean isEnabled() {
        return settings.isEnabled();
    }

    public String createExportFolder() {
        LocalDate exportDate = LocalDate.now();
        YPath monthsDir = settings.getRootExportPath().child(MONTH_FORMATTER.print(exportDate));
        settings.getYt().cypress().create(monthsDir, CypressNodeType.MAP, true, true);

        int counter = 0;
        YPath pathToExport =
                monthsDir.child("Export_" + DATE_FORMATTER.print(exportDate) + "_" + getCounterSuffix(counter++));
        while (settings.getYt().cypress().exists(pathToExport)) {
            pathToExport =
                    monthsDir.child("Export_" + DATE_FORMATTER.print(exportDate) + "_" + (getCounterSuffix(counter++)));
        }

        settings.getYt().cypress().create(pathToExport, CypressNodeType.MAP, true, true);
        return pathToExport.toString();
    }

    public String getCurrentExport() {
        return settings.getYt().cypress().get(settings.getRootExportPath().child("current&").attribute("target_path"))
                .stringValue();
    }

    public boolean isAlreadyExported() {
        Option<Instant> currentExportInstant = getCurrentExportInstant();
        return currentExportInstant.map(d -> d.isAfter(LocalDate.now().toDateTimeAtStartOfDay())).getOrElse(false);
    }

    public void exportCompleted(String exportedFolder) {
        //switch current
        doInTransaction("switching current", transactionId -> {
            YPath currentExportPath = getCurrentExportPath();
            String now = ISODateTimeFormat.basicDateTime().print(Instant.now());
            settings.getYt().cypress().set(transactionId, true, currentExportPath.attribute("ACTIVE_TO"), now);
            settings.getYt().cypress().link(YPath.simple(exportedFolder), currentExportPath, true);
            settings.getYt().cypress().set(transactionId, true, currentExportPath.attribute("ACTIVE_FROM"), now);
        });
    }

    public MapF<String, ListF<String>> findExportFolders() {
        ListF<YTreeStringNode> months = Cf.wrap(settings.getYt().cypress().list(settings.getRootExportPath()));
        months = months.filter(n -> !Objects.equals(n.getValue(), "current"));

        MapF<String, ListF<String>> result = Cf.hashMap();
        for (YTreeStringNode month : months) {
            YPath path = settings.getRootExportPath().child(month.getValue());
            ListF<YTreeStringNode> exportFolders = Cf.wrap(settings.getYt().cypress().list(path));
            if (exportFolders.isEmpty()) {
                continue;
            }

            result.put(path.toString(),
                    Cf.toArrayList(exportFolders.map(name -> path.child(name.getValue()).toString())));
        }

        return result;
    }

    public ListF<String> findEmptyExportTables(String exportFolder) {
        YPath exportPath = YPath.simple(exportFolder);
        ListF<YTreeStringNode> tables = Cf.wrap(settings.getYt().cypress().list(exportPath));
        tables = tables.filter(n -> settings.getYt().getRowCount(exportPath.child(n.getValue())) == 0);
        return tables.map(t -> exportPath.child(t.getValue()).toString()).sorted();
    }

    public void removeExportFolder(String path) {
        settings.getYt().cypress().remove(YPath.simple(path));
    }

    @Override
    public boolean isValidExport(String path) {
        return settings.getYt().cypress().exists(YPath.simple(path).attribute("ACTIVE_FROM"));
    }

    public void overwriteTableInTransaction(String pathToTable,
                                            Function2<Option<UUID>, Integer, ListF<TExportRow>> dataProducer) {
        YPath tablePath = YPath.simple(pathToTable).append(true);
        doInTransaction("export table " + pathToTable, transactionId -> {
            settings.getYt().cypress().remove(transactionId, true, tablePath);
            settings.getYt().cypress().create(transactionId, true, tablePath, CypressNodeType.TABLE, true, false,
                    buildTableAttributes().asMap());

            final int EXPORT_BATCH_SIZE = batchSizeProvider.apply();
            Option<UUID> lastExported = Option.empty();
            int lastPageSize = EXPORT_BATCH_SIZE;
            while (lastPageSize == EXPORT_BATCH_SIZE) {
                ListF<TExportRow> transactions = dataProducer.apply(lastExported, EXPORT_BATCH_SIZE);

                if (transactions.isEmpty()) {
                    break;
                }
                ListF<JsonNode> map = transactions.map(this::mapExportRow);

                settings.getYt().tables().write(transactionId, true, tablePath, YTableEntryTypes.JACKSON_UTF8,
                        map.iterator());

                logger.info("Batch with size {} starting with {} exported to table {} in transaction {}",
                        transactions.size(), lastExported, pathToTable, transactionId);
                lastExported = transactions.lastO().map(YtExportRow::getId);
                lastPageSize = transactions.size();
            }
        });
    }

    protected void doInTransaction(String operationName, Function1V<Optional<GUID>> callback) {
        Optional<GUID> transactionId = Optional.empty();
        try {
            Transaction transaction = settings.getYt().transactions().startAndGet(
                    TRANSACTION_MAX_DURATION);
            transactionId = Option.of(transaction.getId()).toOptional();

            logger.info("Starting {} in transaction {}", operationName, transactionId);
            callback.apply(Optional.of(transaction.getId()));

            settings.getYt().transactions().commit(transaction.getId());
            logger.info("{} in transaction {} completed", operationName, transactionId);
        } catch (Exception e) {
            logger.error("{} in transaction {} failed: {}", operationName, transactionId, e.getMessage(), e);
            transactionId.ifPresent(id -> settings.getYt().transactions().abort(id));
            throw e;
        }
    }

    protected static YTreeBuilder addColumn(YTreeBuilder builder, String name, String type, boolean required) {
        builder.beginMap();

        builder.key("name").value(name);
        builder.key("required").value(required);
        builder.key("type").value(type);

        builder.endMap();
        return builder;
    }

    protected abstract YTreeNode buildTableAttributes();

    protected abstract JsonNode mapExportRow(TExportRow exportRow);

    @Override
    public Option<Instant> getCurrentExportInstant() {
        YPath currentExportPath = getCurrentExportPath();
        YPath modificationAttribute = currentExportPath.attribute("modification_time");
        if (!settings.getYt().cypress().exists(modificationAttribute)) {
            return Option.empty();
        }
        YTreeNode node = settings.getYt().cypress().get(modificationAttribute);
        return Option.ofNullable(
                ISO_FORMAT.parseDateTime(node.stringNode().getValue()).toInstant());
    }

    private YPath getCurrentExportPath() {
        return settings.getRootExportPath().child("current");
    }

    private String getCounterSuffix(int counter) {
        return String.format("%02d", counter);
    }
}
