package ru.yandex.chemodan.app.psbilling.core.billing.groups.export.distributionplatform;

import com.google.common.annotations.VisibleForTesting;
import lombok.Setter;
import org.joda.time.LocalDate;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.BaseExportService;
import ru.yandex.chemodan.app.psbilling.core.config.YtExportSettings;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.DistributionServiceTransactionCalculation;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DistributionPlatformTransactionsExportService
        extends BaseExportService<DistributionPlatformTransactionsDao.ExportRow,
        YtOperationsDistributionPlatformTransactions> {
    private static final Logger logger = LoggerFactory.getLogger(BaseExportService.class);

    @VisibleForTesting
    @Setter
    private DynamicProperty<Integer> exportMonthsCount =
            new DynamicProperty<>("ps-billing.distribution_platform.export.months_count", 2);

    private final DistributionPlatformTransactionsDao distributionPlatformTransactionsDao;
    private final DistributionPlatformCalculationDao distributionPlatformCalculationDao;
    private final TransactionTemplate transactionTemplate;

    public DistributionPlatformTransactionsExportService(
            BazingaTaskManager bazingaTaskManager,
            YtExportSettings primaryYtExportSetting,
            YtExportSettings secondaryYtExportSetting,
            DistributionPlatformTransactionsDao distributionPlatformTransactionsDao,
            DistributionPlatformCalculationDao distributionPlatformCalculationDao,
            TransactionTemplate transactionTemplate,
            Function0<Integer> batchSizeProvider) {
        super(bazingaTaskManager, new YtOperationsDistributionPlatformTransactionsImpl(primaryYtExportSetting,
                        batchSizeProvider),
                new YtOperationsDistributionPlatformTransactionsImpl(secondaryYtExportSetting, batchSizeProvider));
        this.distributionPlatformTransactionsDao = distributionPlatformTransactionsDao;
        this.distributionPlatformCalculationDao = distributionPlatformCalculationDao;
        this.transactionTemplate = transactionTemplate;
    }

    @Override
    public void scheduleExport() {
        export();
    }

    @Override
    // в текущем месяце чистим все пустые данные кроме последнего экспорта
    // в остальных оставляем по 1 последнему экспорту
    public void removeOldExports(YtOperationsDistributionPlatformTransactions exporter) {
        MapF<String, ListF<String>> exportFoldersMap = exporter.findExportFolders();

        boolean lastMonth = true;
        for (Tuple2<String, ListF<String>> tuple2 : exportFoldersMap.entries().sortedBy1Desc()) {
            ListF<String> exports = tuple2._2.sorted().reverse();
            boolean lastExport = true;
            for (String export : exports) {
                boolean validExport = exporter.isValidExport(export);
                if (lastExport && validExport) {
                    logger.info("Saving last export in month {}: {}", tuple2._1, export);
                    lastExport = false;
                    continue;
                }

                if (lastMonth) {
                    ListF<String> emptyExportTables = exporter.findEmptyExportTables(export);
                    logger.info("Removing empty exports {};", emptyExportTables);
                    emptyExportTables.forEach(exporter::removeExportFolder);
                } else {
                    logger.info("Removing export {}; was valid = {}", export, validExport);
                    exporter.removeExportFolder(export);
                }
            }
            lastMonth = false;
        }
    }

    @Override
    protected void doExport(YtOperationsDistributionPlatformTransactions exporter, boolean force) {
        String exportFolder = exporter.createExportFolder();
        for (int monthShift = 1; monthShift <= exportMonthsCount.get(); monthShift++) {
            LocalDate calcMonth = LocalDate.now().minusMonths(monthShift).withDayOfMonth(1);
            boolean exportSuccess = exportMonth(calcMonth, exporter, exportFolder);
            if (!exportSuccess) {
                logger.error("got tasks not in terminal state for month: {}", calcMonth);
                return;
            }
        }

        exporter.exportCompleted(exportFolder);
        logger.info("Export completed");
    }

    private boolean doWithLockedCalculation(LocalDate calcMonth,
                                            Function1V<ListF<DistributionServiceTransactionCalculation>> callback) {
        return transactionTemplate.execute(status -> {
            ListF<DistributionServiceTransactionCalculation> calculations =
                    distributionPlatformCalculationDao.lock(calcMonth);
            if (calculations.stream().anyMatch(x -> x.getStatus() != CalculationStatus.COMPLETED)) {
                logger.warn("got tasks not in terminal state for month: {}", calcMonth);
                return false;
            }
            callback.apply(calculations);
            return true;
        });
    }

    private boolean exportMonth(LocalDate calcMonth, YtOperationsDistributionPlatformTransactions exporter,
                                String exportFolder) {
        return doWithLockedCalculation(calcMonth, calculations -> {
            LocalDate exportMonth = calcMonth.plusMonths(1);
            LocalDate exportDate = exportMonth.withDayOfMonth(1);
            fillMonthExceptDate(exporter, exportFolder, exportMonth, exportDate);
            exporter.overwriteTableInTransaction(buildTablePathForDate(exportFolder, exportDate),
                    (from, batchSize) -> distributionPlatformTransactionsDao.findExportRows(calcMonth));
        });
    }

    //  нужно для корректной работы Платформы Дистрибуции https://st.yandex-team.ru/CHEMODAN-82951#62442d58dadba84a4c78600a
    private void fillMonthExceptDate(YtOperationsDistributionPlatformTransactions exporter, String exportFolder,
                                     LocalDate month, LocalDate except) {
        int lastDay = month.dayOfMonth().getMaximumValue();
        for (int day = 1; day <= lastDay; day++) {
            LocalDate curDate = month.withDayOfMonth(day);
            if (curDate.equals(except)) {
                continue;
            }

            exporter.overwriteTableInTransaction(buildTablePathForDate(exportFolder, curDate),
                    (from, batchSize) -> Cf.list());
        }
    }
}
