package ru.yandex.chemodan.app.psbilling.core.billing.groups.export.groupservices;


import java.util.Objects;

import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.export.BaseExportService;
import ru.yandex.chemodan.app.psbilling.core.config.YtExportSettings;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.COMPLETED;
import static ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus.DUMPING;

public class GroupServiceTransactionsExportService
        extends BaseExportService<GroupServiceTransactionsDao.ExportRow, YtOperationsGroupTransactions> {
    private static final Logger logger = LoggerFactory.getLogger(GroupServiceTransactionsExportService.class);

    private final GroupServiceTransactionsDao groupServiceTransactionsDao;
    private final GroupServiceTransactionCalculationDao groupServiceTransactionCalculationDao;
    private final BazingaTaskManager bazingaTaskManager;
    private final DynamicProperty<Integer> exportsToKeep =
            new DynamicProperty<>("transactions.exports.keep.count", 30);

    public GroupServiceTransactionsExportService(Integer serviceId,
                                                 GroupServiceTransactionsDao groupServiceTransactionsDao,
                                                 GroupServiceTransactionCalculationDao groupServiceTransactionCalculationDao,
                                                 BazingaTaskManager bazingaTaskManager,
                                                 YtExportSettings primaryYtExportSettings,
                                                 YtExportSettings secondaryYtExportSettings,
                                                 Function0<Integer> batchSizeProvider) {
        super(bazingaTaskManager, new YtOperationsGroupTransactionsImpl(primaryYtExportSettings, serviceId, batchSizeProvider),
                new YtOperationsGroupTransactionsImpl(secondaryYtExportSettings, serviceId, batchSizeProvider));
        this.groupServiceTransactionsDao = groupServiceTransactionsDao;
        this.groupServiceTransactionCalculationDao = groupServiceTransactionCalculationDao;
        this.bazingaTaskManager = bazingaTaskManager;
    }

    public void forceExportLatestCalulcations() throws Exception {
        LocalDate latestCalculation = groupServiceTransactionCalculationDao.findLatestCalculation();
        doForTwoClusters("Export", operations -> export2MonthsBefore(operations, latestCalculation, true));
    }

    @Override
    public void scheduleExport() {
        bazingaTaskManager.schedule(new GroupServiceTransactionsExportTask());
    }

    @Override
    // оставляем первые N экспортов, но гарантированно в каждой папке оставляем хотя бы один экспорт
    protected void removeOldExports(YtOperationsGroupTransactions exporter) {
        MapF<String, ListF<String>> exportFoldersMap = exporter.findExportFolders();
        String currentExportPath = exporter.getCurrentExport();

        int storedExportsCnt = 0;
        for (Tuple2<String, ListF<String>> tuple2 : exportFoldersMap.entries().sortedBy1Desc()) {
            ListF<String> exports = tuple2._2.sorted().reverse();

            boolean firstProcessed = false;
            for (String export : exports) {
                if (Objects.equals(export, currentExportPath)) {
                    logger.info("Saving export {} because it's current", export);
                    continue;
                }

                if (!firstProcessed && exporter.isValidExport(export)) {
                    logger.info("Saving last export in month {}: {}", tuple2._1, export);
                    firstProcessed = true;
                    continue;
                }

                if (storedExportsCnt < exportsToKeep.get()) {
                    logger.info("Storing export {} because of it's fresh enough", export);
                    storedExportsCnt++;
                } else {
                    logger.info("Removing export {}", export);
                    exporter.removeExportFolder(export);
                }
            }
        }
    }

    @Override
    protected void doExport(YtOperationsGroupTransactions exporter, boolean force) {
        LocalDate now = LocalDate.now();
        LocalDate yesterday = now.minusDays(1);

        export2MonthsBefore(exporter, yesterday, force);
    }

    void export2MonthsBefore(YtOperationsGroupTransactions exporter, LocalDate date, boolean force) {
        if (!force && exporter.isAlreadyExported()) {
            logger.info("already exported to cluster");
            return;
        }

        //lock yesterday calculation on whole export time
        doWithLockedCalculation(date,
                "Calculation on day {} doesn't ready for export for some reason, skipping whole export",
                () -> {
                    String exportFolder = exporter.createExportFolder();
                    LocalDate current = date.minusMonths(2);
                    //yesterday will be the last
                    while (!current.isAfter(date)) {
                        LocalDate currentFinal = current;
                        RetryUtils.retry(logger, 5, 5000, 2, () -> {
                            exportDay(exporter, buildTablePathForDate(exportFolder, currentFinal), currentFinal);
                            return null;
                        });
                        current = current.plusDays(1);
                    }

                    exporter.exportCompleted(exportFolder);
                    logger.info("Export completed");
                }
        );
    }

    void exportLast2Months(YtOperationsGroupTransactions exporter, boolean force) {
        LocalDate now = LocalDate.now();
        LocalDate yesterday = now.minusDays(1);

        export2MonthsBefore(exporter, yesterday, force);
    }

    private void exportDay(YtOperationsGroupTransactions exporter, String tablePath, LocalDate date) {
        // if calculation on day not ready, skip export
        doWithLockedCalculation(date,
                "Calculation on day {} doesn't exist, or in status, differ from completed or dumping, export will be " +
                        "skipped",
                () -> exporter.overwriteTableInTransaction(
                        tablePath, (uuidO, count) -> groupServiceTransactionsDao.findTransactions(date, uuidO, count)
                )
        );
    }

    private void doWithLockedCalculation(LocalDate date, String lockFailedMessage, Function0V callback) {
        boolean updated =
                groupServiceTransactionCalculationDao.updateStatus(date, Cf.list(COMPLETED, DUMPING), DUMPING);
        if (!updated) {
            logger.warn(lockFailedMessage, date);
            return;
        }

        try {
            callback.apply();
        } finally {
            RetryUtils.retry(logger, 5, 5000, 2, () -> {
                groupServiceTransactionCalculationDao.updateStatus(date, Cf.list(COMPLETED, DUMPING), COMPLETED);
                return null;
            });
        }
    }
}
