package ru.yandex.chemodan.app.psbilling.core.billing.groups.payment;

import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks.CheckCardBindingStatusTask;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.TrustCardBindingDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardBindingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.TrustCardBinding;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.chemodan.balanceclient.model.response.CheckBindingResponse;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

@AllArgsConstructor
public class CardBindingChecker {
    private static final Logger logger = LoggerFactory.getLogger(CardBindingChecker.class);
    private static final int CARD_BINDINGS_FETCH_BATCH_SIZE = 1000;

    private final BalanceService balanceService;
    private final GroupBillingService groupBillingService;
    private final CardDao cardDao;
    private final GroupsManager groupsManager;
    @SuppressWarnings("unused")
    private final FeatureFlags featureFlags;
    private final TaskScheduler taskScheduler;
    private final TrustCardBindingDao trustCardBindingDao;

    // проверяем только привязки, которые созданы до now().minus(checkCardBindingsIntervalEnd)
    private final DynamicProperty<Integer> checkCardBindingsIntervalEnd =
            new DynamicProperty<>("ps-billing-check-card-bindings-interval-end-minutes", 1);

    // в случае успешной привязки карты включаем автосписание только в течение
    // canEnableAutoBillingAfterCardBindingInterval
    private final DynamicProperty<Integer> canEnableAutoBillingAfterCardBindingInterval =
            new DynamicProperty<>("ps-billing-can-enable-autobilling-after-card-binding-interval-minutes", 1440);
    // если за cardBindingTimeoutForErrorStatus не получили error и success от траста, считаем, что привязка error
    private final DynamicProperty<Integer> cardBindingTimeoutForErrorStatus =
            new DynamicProperty<>("ps-billing-card-binding-timeout-for-error-status-minutes", 60);

    // если получили от траста id карты new_card, то ретраим в цикле retryCardBindingTimesOnNewCard раз с промежутком
    // retryCardBindingIntervalOnNewCard
    private final DynamicProperty<Integer> retryCardBindingIntervalOnNewCard =
            new DynamicProperty<>("ps-billing-retry-card-binding-interval-on-new-card-milliseconds", 500);
    private final DynamicProperty<Integer> retryCardBindingTimesOnNewCard =
            new DynamicProperty<>("ps-billing-retry-card-binding-times-on-new-card", 4);

    public void scheduleToCheckInitBindings() {
        Instant createdBefore = Instant.now().minus(Duration.standardMinutes(checkCardBindingsIntervalEnd.get()));
        ListF<TrustCardBinding> bindingsToCheck =
                BatchFetchingUtils.collectBatchedEntities(
                        (batchSize, id) -> trustCardBindingDao
                                .findInitBindings(createdBefore, batchSize, id),
                        TrustCardBinding::getId,
                        CARD_BINDINGS_FETCH_BATCH_SIZE,
                        logger
                );

        bindingsToCheck
                .map(binding -> new CheckCardBindingStatusTask.Parameters(binding.getId()))
                .map(CheckCardBindingStatusTask::new)
                .forEach(taskScheduler::schedule);
    }

    public void checkCardBindingStatus(UUID bindingId) {
        Option<TrustCardBinding> bindingO = trustCardBindingDao.findByIdO(bindingId);
        if (bindingO.isEmpty()) {
            logger.error("card binding {} is not found", bindingId.toString());
            return;
        }

        TrustCardBinding binding = bindingO.get();

        if (binding.getStatus() == CardBindingStatus.SUCCESS) {
            logger.info("ignore checking card binding {}: " +
                    "already in success status", bindingId.toString());
            return;
        }

        if (binding.getTransactionId().isEmpty()) {
            if (binding.getCreatedAt().plus(Duration.standardMinutes(
                    cardBindingTimeoutForErrorStatus.get())).isBeforeNow()) {
                logger.warn("card binding {} transactionId is not found: " +
                        "set error status due to timeout", bindingId.toString());
                trustCardBindingDao.updateStatusIfInNotSuccess(binding.withStatus(CardBindingStatus.ERROR)
                        .withError(Option.of("timeout, empty transactionId")));
            }
            else {
                logger.warn("card binding {} " +
                        "transactionId is not found", bindingId.toString());
            }
            return;
        }

        Option<BalancePaymentInfo> paymentInfo = groupsManager.getPaymentInfo(binding.getOperatorUid());

        CardBindingResult bindingResult = checkBindingAndSaveCard(
                binding.getOperatorUid(), binding.getTransactionId().get(),
                paymentInfo, binding.getCreatedAt());
        logger.info("card binding {} is in {} status",
                bindingId.toString(), bindingResult.getBindingStatus());

        int currentRetryNo = 0;
        while (currentRetryNo < retryCardBindingTimesOnNewCard.get() &&
                bindingResult.getCardExternalId().isPresent() &&
                bindingResult.getCardExternalId().get().equals("new_card")) {
            // костыль для CHEMODAN-82878, когда Баланс возвращает id карты new_card,
            // привязка скорее всего вот-вот завершится - ретраим проверку
            ThreadUtils.sleep(retryCardBindingIntervalOnNewCard.get());
            bindingResult = checkBindingAndSaveCard(
                    binding.getOperatorUid(), binding.getTransactionId().get(), paymentInfo, binding.getCreatedAt());
            ++currentRetryNo;
        }

        Option<String> errorDesc = bindingResult.getError();
        CardBindingStatus newStatus;
        if (bindingResult.getCardId().isPresent()) {
            newStatus = CardBindingStatus.SUCCESS;
        }
        else if (bindingResult.getBindingStatus().equals(CheckBindingResponse.ERROR_STATUS)) {
            newStatus = CardBindingStatus.ERROR;
        }
        else {
            if (binding.getCreatedAt().plus(Duration.standardMinutes(
                    cardBindingTimeoutForErrorStatus.get())).isBeforeNow()) {
                logger.warn("card binding {} is in {} status: set error status due to timeout",
                        bindingId.toString(), bindingResult.getBindingStatus());
                newStatus = CardBindingStatus.ERROR;
                String errorDescWithTimeout = "timeout";
                if (errorDesc.isPresent() && !errorDesc.get().isEmpty()) {
                    errorDescWithTimeout += (", " + errorDesc.get());
                }
                errorDesc = Option.of(errorDescWithTimeout);
            }
            else {
                return;
            }
        }

        trustCardBindingDao.updateStatusIfInNotSuccess(binding.withStatus(newStatus)
                .withError(errorDesc).withCardId(bindingResult.getCardId()));
    }

    public CardBindingResult checkBindingAndSaveCard(PassportUid operatorUid,
                                                      String transactionId,
                                                      Option<BalancePaymentInfo> paymentInfo,
                                                      Instant operationDate) {
        boolean canEnableAutoBilling = operationDate.plus(Duration.standardMinutes(
                canEnableAutoBillingAfterCardBindingInterval.get())).isAfterNow();
        CheckBindingResponse bindingResponse = balanceService.checkBinding(operatorUid.getUid(), transactionId);
        if (!bindingResponse.getBindingResult().equals(CheckBindingResponse.SUCCESS_BINDING_STATUS)) {
            logger.info("balance checkBinding returned status {} for {}",
                    bindingResponse.getBindingResult(), operatorUid.toString());
            return new CardBindingResult(bindingResponse.getBindingResult(),
                    Option.empty(), Option.empty(),
                    bindingResponse.getPaymentResponseDescription());
        }

        Option<String> cardExternalId = bindingResponse.getPaymentMethodId();
        if (cardExternalId.isEmpty()) {
            logger.warn("balance checkBinding returned no paymentMethodId with status success for {}",
                    operatorUid.toString());
            return new CardBindingResult(bindingResponse.getBindingResult(),
                    Option.empty(), Option.empty(),
                    bindingResponse.getPaymentResponseDescription());
        }

        if (cardExternalId.get().equals("new_card")) {
            return new CardBindingResult(bindingResponse.getBindingResult(),
                    Option.empty(), cardExternalId,
                    bindingResponse.getPaymentResponseDescription());
        }

        boolean needToEnableAutoBilling = canEnableAutoBilling &&
                paymentInfo.map(p -> !p.isB2bAutoBillingEnabled()).orElse(false);

        CardEntity card = (needToEnableAutoBilling ?
                cardDao.setB2BPrimaryCard(operatorUid, cardExternalId.get()) :
                cardDao.insertOrUpdateStatusWithCheckingOtherPrimary(CardDao.InsertData.builder()
                        .uid(operatorUid)
                        .purpose(CardPurpose.B2B)
                        .externalId(cardExternalId.get())
                        .status(CardStatus.ACTIVE).build())
        );

        if (needToEnableAutoBilling) {
            groupBillingService.setAutoBillingOn(paymentInfo.get(), true);
        }

        return new CardBindingResult(bindingResponse.getBindingResult(),
                Option.of(card.getId()), cardExternalId, Option.empty());
    }

    @Data
    public static class CardBindingResult {
        private final String bindingStatus;
        private final Option<UUID> cardId;
        private final Option<String> cardExternalId;
        private final Option<String> error;
    }
}
